/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.metadata.type;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.mule.metadata.api.utils.MetadataTypeUtils.getTypeId;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.internal.metadata.type.TypeRepository;

import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Helper class to resolve metadata type operations.
 *
 * @since 1.1
 */
public class MetadataTypeHelper {

  private static final Logger LOGGER = LoggerFactory.getLogger(MetadataTypeHelper.class);

  private MetadataTypeHelper() {}

  /**
   * Given a set of {@link ExtensionModel extensionModels} it will collect those base metadata types from the context that have
   * been contributed with subTypes.
   *
   * @param extensions context with the set of {@link ExtensionModel extensionModels} to resolve the types hierarchy.
   * @return a {@link List} of {@link ObjectTypeHierarchy} with those metadata types from the context that have
   * been contributed with subTypes.
   */
  public static List<ObjectTypeHierarchy> getObjectTypeHierarchies(Set<ExtensionModel> extensions) {
    TypeRepository typeRepository = new TypeRepository(extensions);
    return typeRepository.getAllBaseTypes().stream().map(baseType -> {
      ObjectTypeHierarchy objectTypeHierarchy = new ObjectTypeHierarchy();
      objectTypeHierarchy.setType(baseType);
      Optional<String> declaringExtension = typeRepository.getDeclaringExtension(baseType);
      if (declaringExtension.isPresent()) {
        objectTypeHierarchy.setExtensionName(declaringExtension.get());
        objectTypeHierarchy.setSubTypeMappings(typeRepository.getSubTypes(baseType)
            .stream()
            .map(subType -> {
              SubTypeMapping subTypeMapping = new SubTypeMapping();
              subTypeMapping.setObjectType(subType);
              subTypeMapping
                  .setExtensionName(typeRepository.getDeclaringExtension(subType).get());
              return subTypeMapping;
            })
            .collect(toSet()));
      } else {
        LOGGER.warn(String.format("Ignoring baseType: '%s' as it is not defined by an extension in its type catalog",
                                  getTypeId(baseType).orElse(baseType.toString())));
      }
      return objectTypeHierarchy;
    }).filter(objectTypeHierarchy -> !objectTypeHierarchy.getSubTypeMappings().isEmpty()).collect(toList());
  }

}
