/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.serialization;

import static com.google.common.collect.ImmutableMap.of;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.collection.IsMapContaining.hasEntry;
import static org.junit.Assert.assertThat;
import static org.mule.metadata.api.model.MetadataFormat.JAVA;
import org.mule.metadata.api.annotation.MetadataFormatPropertiesAnnotation;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.model.StringType;

import java.util.Arrays;
import java.util.Collection;
import java.util.Optional;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(Parameterized.class)
public class TypeAnnotationMapperTestCase {

  private static final String VALUE = "someValue";

  private final Serializer serializer;

  @Parameterized.Parameters(name = "{0}")
  public static Collection<Object[]> data() {
    return Arrays.asList(new Object[][] {
        {"xStream", new XStreamClientSerializer()},
        {"Kryo", new KryoClientSerializer()}
    });
  }

  public TypeAnnotationMapperTestCase(String name, Serializer serializer) {
    this.serializer = serializer;
  }

  @Test
  public void typeAnnotationNotMapped() {
    StringType stringType = new BaseTypeBuilder(JAVA)
        .stringType()
        .with(new TestMetadataFormatPropertiesAnnotation(VALUE))
        .build();

    String content = serializer.serialize(stringType);
    StringType deserialized = serializer.deserialize(content);

    assertThat(deserialized.getAnnotation(TestMetadataFormatPropertiesAnnotation.class).isPresent(), is(false));
  }

  @Test
  public void mappingForMetadataFormatPropertiesAnnotation() {
    StringType stringType = new BaseTypeBuilder(JAVA)
        .stringType()
        .with(new MetadataFormatPropertiesAnnotation(of("key", "value")))
        .build();

    String content = serializer.serialize(stringType);
    StringType deserialized = serializer.deserialize(content);

    Optional<MetadataFormatPropertiesAnnotation> annotation =
        deserialized.getAnnotation(MetadataFormatPropertiesAnnotation.class);
    assertThat(annotation.isPresent(), is(true));
    assertThat(annotation.get().getValue(), hasEntry("key", "value"));

  }


}
