/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.configuration.agent;

import static java.util.Objects.requireNonNull;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.configuration.agent.proxy.ProxyConfig;
import org.mule.tooling.client.api.configuration.ssl.SslConfiguration;
import org.mule.tooling.client.internal.configuration.agent.ImmutableAgentConfiguration;

import java.net.URL;
import java.util.Optional;

/**
 * Configuration settings for a {@link ToolingRuntimeClient}.
 *
 * @since 1.0
 */
public interface AgentConfiguration {

  /**
   * @return {@link URL} for the REST Tooling service API.
   */
  URL getToolingApiUrl();

  /**
   * @return default timeout in milliseconds for connection on REST client.
   */
  long getDefaultConnectTimeout();

  /**
   * @return default timeout in milliseconds for read on REST client.
   */
  long getDefaultReadTimeout();

  /**
   * @return {@link SslConfiguration} for the REST Tooling service API to configure the client.
   */
  Optional<SslConfiguration> getSslConfiguration();

  /**
   * @return {@link ProxyConfig} for the REST Tooling service API to configure the client.
   */
  Optional<ProxyConfig> getProxyConfig();

  /**
   * @return a {@link Builder} to build a {@link AgentConfiguration}.
   */
  static Builder builder() {
    return new Builder();
  }

  class Builder {

    private URL toolingApiUrl;
    private long defaultConnectionTimeout = 0;
    private long defaultReadTimeout = 0;
    private SslConfiguration sslConfiguration = null;
    private ProxyConfig proxyConfig = null;

    /**
     * Sets the {@link URL} for the remote REST tooling service used by {@link ToolingRuntimeClient}.
     *
     * @param toolingApiUrl {@link URL} for the remote REST tooling service used by {@link ToolingRuntimeClient}.
     * @return this
     */
    public Builder withToolingApiUrl(URL toolingApiUrl) {
      requireNonNull(toolingApiUrl, "toolingApiUrl cannot be null");
      this.toolingApiUrl = toolingApiUrl;
      return this;
    }

    /**
     * Sets the default connection timeout in milliseconds for the REST client. Default value is 0 (infinity).
     *
     * @param defaultConnectionTimeout milliseconds to set the default connection timeout.
     * @return this
     */
    public Builder withDefaultConnectionTimeout(long defaultConnectionTimeout) {
      checkArgument(defaultConnectionTimeout >= 0, "defaultConnectionTimeout cannot be < 0");
      this.defaultConnectionTimeout = defaultConnectionTimeout;
      return this;
    }

    /**
     * Sets the default read timeout in milliseconds for the REST client. Default value is 0 (infinity).
     *
     * @param defaultReadTimeout milliseconds to set the default read timeout.
     * @return this
     */
    public Builder withDefaultReadTimeout(long defaultReadTimeout) {
      checkArgument(defaultReadTimeout >= 0, "defaultReadTimeout cannot be < 0");
      this.defaultReadTimeout = defaultReadTimeout;
      return this;
    }

    /**
     * Sets the SSLContext
     *
     * @param sslConfiguration {@link SslConfiguration} to set for the REST Client.
     * @return this
     */
    public Builder withSslConfiguration(SslConfiguration sslConfiguration) {
      this.sslConfiguration = sslConfiguration;
      return this;
    }

    /**
     * Sets the ProxyConfig
     *
     * @param proxyConfig {@link ProxyConfig} to set for the REST Client.
     * @return this
     */
    public Builder withProxyConfig(ProxyConfig proxyConfig) {
      this.proxyConfig = proxyConfig;
      return this;
    }

    /**
     * Builds the {@link AgentConfiguration} object.
     *
     * @return {@link AgentConfiguration} with the value sets.
     */
    public AgentConfiguration build() {
      return new ImmutableAgentConfiguration(this.toolingApiUrl, this.defaultConnectionTimeout, this.defaultReadTimeout,
                                             sslConfiguration, proxyConfig);
    }

    protected static void checkArgument(boolean expression, Object errorMessage) {
      if (!expression) {
        throw new IllegalArgumentException(String.valueOf(errorMessage));
      }
    }

  }

}
