/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationService;
import org.mule.tooling.client.api.artifact.dsl.DslElementSyntax;
import org.mule.tooling.client.api.artifact.dsl.DslSyntaxResolverService;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.api.datasense.MetadataCacheFactory;
import org.mule.tooling.client.api.exception.ToolingArtifactNotFoundException;
import org.mule.tooling.client.api.icons.ExtensionIconsService;
import org.mule.tooling.client.api.extension.ExtensionModelService;
import org.mule.tooling.client.api.message.history.MessageHistoryService;

import java.net.URL;
import java.util.Map;

/**
 * Provides tooling services for Mule Runtime.
 *
 * @since 1.0
 */
public interface ToolingRuntimeClient {

  interface Builder {

    /**
     * Sets the {@link AgentConfiguration} used by the client to access the Tooling REST API. If the operations executed by the
     * client don't require to access Tooling REST API this won't need to be set. Otherwise an operation could throw a
     * {@link org.mule.tooling.client.api.exception.MissingToolingConfigurationException} in case if there was no configuration
     * set for the client instance.
     *
     * @param agentConfiguration {@link AgentConfiguration} for the client.
     * @return this
     */
    Builder withRemoteAgentConfiguration(AgentConfiguration agentConfiguration);

    /**
     * Sets the {@link MavenConfiguration} used by the client to access the maven artifacts.
     *
     * If not provided, then the bootstrap maven configuration will be used.
     *
     * @param mavenConfiguration {@link MavenConfiguration} for the client.
     * @return this
     */
    Builder withMavenConfiguration(MavenConfiguration mavenConfiguration);

    /**
     * Sets the {@link MetadataCacheFactory} used by the client to create {@link org.mule.tooling.client.api.datasense.MetadataCache}.
     *
     * If not provided, metadata resolution will be always fetched from Mule Runtime when resolving data sense.
     *
     * @param metadataCacheFactory {@link MetadataCacheFactory} factory for {@link org.mule.tooling.client.api.datasense.MetadataCache}.
     * @return this.
     */
    Builder withMetadataCacheFactory(MetadataCacheFactory metadataCacheFactory);

    /**
     * Builds the {@link ToolingRuntimeClient}.
     *
     * @return a new instance of the {@link ToolingRuntimeClient}.
     */
    ToolingRuntimeClient build();

  }

  /**
   * Creates a {@link ToolingArtifact} for the application from the URL of the application content. It can be a file protocol
   * pointing to a the expanded content of the application jar or a remote location pointing to the jar file where it would be read.
   * <p/>
   * It is expected that users of this API should handle the packaging for the application content, therefore if {@link ToolingArtifact ToolingArtifacts}
   * are created per request keeping the same applicationUrlContent when there is no need to package again would avoid wasting time on
   * packaging when not needed.
   *
   * @param applicationUrlContent {@link URL} for the application content.
   * @param toolingArtifactProperties {@link Map} of properties for the {@link org.mule.tooling.client.api.artifact.ToolingArtifact}. Useful when
   * need to keep context for tracking {@link ToolingArtifact ToolingArtifacts}, for instance: projectId to associated these
   * {@link ToolingArtifact} to a project and when fetching Metadata from {@link org.mule.tooling.client.api.datasense.MetadataCache} get the projectId.
   * @return a {@link ToolingArtifact}.
   */
  ToolingArtifact newToolingArtifact(URL applicationUrlContent, Map<String, String> toolingArtifactProperties);

  /**
   * Fetches a {@link ToolingArtifact} already created to resolve operations over an application.
   *
   * @param id the identifier for the {@link ToolingArtifact} to be recovered.
   * @return a {@link ToolingArtifact}
   * @throws ToolingArtifactNotFoundException if the {@link ToolingArtifact} is no longer present.
   */
  ToolingArtifact fetchToolingArtifact(String id) throws ToolingArtifactNotFoundException;

  /**
   * Returns a {@link ExtensionModelService} to allow retrieving the {@link org.mule.tooling.client.api.extension.model.ExtensionModel} of a
   * plugin.
   *
   * @return a {@link ExtensionModelService}
   */
  ExtensionModelService extensionModelService();

  /**
   * Returns a {@link ArtifactSerializationService} to allow serialize and deserialze a Mule artifact declaration.
   *
   * @return a {@link ArtifactSerializationService}
   */
  ArtifactSerializationService artifactSerializationService();

  /**
  * Returns an {@link DslSyntaxResolverService} that allows to resolve the {@link DslElementSyntax} of an extension components
   *
   * @return a {@link DslSyntaxResolverService}
   */
  DslSyntaxResolverService dslSyntaxResolverService();

  /**
   * Returns a {@link MessageHistoryService} to do try it over an application.
   *
   * @return a {@link MessageHistoryService}
   */
  MessageHistoryService messageHistoryService();

  /**
   * Returns a {@link ExtensionIconsService} that allows to retrieve the icons for a given Extension.
   *
   * @return a {@link ExtensionIconsService}
   *
   * @since 1.1
   */
  ExtensionIconsService iconsService();

}
