/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.message.history;

import org.mule.tooling.client.api.exception.NoSuchApplicationException;

import java.util.List;

/**
 * Service to collect message history over an application. The service will allow to enable the collection of transactions over an application,
 * retrieve the data and disable the collection of data.
 * <p/>
 * Applications should be deployed by Mule Agent in order to be enabled for message history.
 * <p/>
 * It is up to the clients of the service to disable the message history, if not the message history job on Mule Agent will dispose the resources
 * and stop the collection on Mule Agent side if no messages are consumes after a certain threshold (configured on Mule Agent side).
 * <p/>
 * The service works as a client for the Mule Agent that buffers the components notifications and each invocation to
 * {@link #consume(String, int)} will update a heartbeat in Mule Agent.
 *
 * @since 1.0
 */
public interface MessageHistoryService {

  /**
   * Enables the message history functionality for the given application. Once enable it, the {@link #consume(String, int)} method would
   * retrieve the transactions collected.
   *
   * @param applicationName a name to be used when deploying the application for collecting notifications
   * @throws NoSuchApplicationException if the application to be enable for try it is not already deployed.
   */
  void enable(String applicationName) throws NoSuchApplicationException;

  /**
   * Consumes the transactions that were buffered by the Mule Agent until this moment for the given application.
   *
   * @param applicationName the application name to get its notifications
   * @param chunkSize the maximum amount of notifications to be returned
   * @return {@link List} of {@link Notification notifications}
   */
  MessageHistory consume(String applicationName, int chunkSize);

  /**
   * Disables the message history functionality for the given application.
   *
   * @param applicationName the application name to be disabled
   */
  void disable(String applicationName);

}
