/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.test.utils;

import static java.lang.Integer.parseInt;
import static java.net.InetAddress.getLocalHost;
import static java.util.Arrays.asList;
import static java.util.Objects.requireNonNull;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static org.awaitility.Awaitility.with;
import org.mule.tck.junit4.rule.FreePortFinder;
import org.mule.tooling.runtime.process.controller.MuleProcessController;
import org.mule.tooling.runtime.process.controller.MuleProcessControllerFactory;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.net.UnknownHostException;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.net.ssl.SSLHandshakeException;

import org.apache.commons.lang3.StringUtils;
import org.awaitility.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Controller to start/stop Mule Runtime in order to allow testing Tooling client operations.
 *
 * @since 1.0
 */
public class MuleStandaloneController {

  public static final String MULE_JAVA_PREFIX_PARAM = "-M-D";
  public static final String EQUALS_CONST = "=";
  public static final String MULE_TOOLING_RESOURCE = "/mule/tooling";
  private static final String HTTPS_PROTOCOL = "https";
  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private final static String MULE_AGENT_REST_TRANSPORT_PORT_PROPERTY = "-M-Drest.agent.transport.port";

  private MuleStandaloneConfiguration configuration;
  private MuleProcessController muleProcessController;
  private AtomicBoolean initialized = new AtomicBoolean(false);

  public MuleStandaloneController(File muleHome, MuleStandaloneConfiguration configuration) {
    requireNonNull(muleHome, "muleHome cannot be null");
    requireNonNull(configuration, "configuration cannot be null");

    this.configuration = configuration;
    this.muleProcessController =
        MuleProcessControllerFactory.createController(muleHome, configuration.getControllerOperationTimeout());
  }

  public void start(String[] args, String protocol) {
    if (initialized.compareAndSet(false, true)) {
      try {
        if (logger.isDebugEnabled()) {
          logger.debug("Starting client");
        }

        List<String> parameters = asList(args);
        parameters.removeIf(StringUtils::isBlank);
        int restAgentPort;
        Optional<String> agentPortParameter = findAgentPortParameter(parameters);

        if (!agentPortParameter.isPresent()) {
          restAgentPort = new FreePortFinder(9000, 3000).find();
          if (logger.isDebugEnabled()) {
            logger.debug("Found a free port for Mule Agent REST at: {}", restAgentPort);
          }
          parameters.add(MULE_AGENT_REST_TRANSPORT_PORT_PROPERTY + EQUALS_CONST + restAgentPort);
        } else {
          restAgentPort = parseInt(agentPortParameter.get().split("=")[1]);
          if (logger.isDebugEnabled()) {
            logger.debug("Port for Mule Agent REST already defined as parameter: {}", restAgentPort);
          }
        }
        muleProcessController.start(parameters.toArray(new String[0]));

        waitUnilRuntimeToolingServiceIsOperational(getToolingApiUrl(protocol, restAgentPort));
      } catch (Exception e) {
        stop();
        throw new IllegalStateException("Error while starting client", e);
      }
    } else {
      logger.warn("Client already started");
    }
  }

  private Optional<String> findAgentPortParameter(List<String> parameters) {
    return parameters.stream().filter(item -> item.startsWith(MULE_JAVA_PREFIX_PARAM)
        && item.split(EQUALS_CONST)[0].equals(MULE_AGENT_REST_TRANSPORT_PORT_PROPERTY)).findFirst();
  }

  private void waitUnilRuntimeToolingServiceIsOperational(URL toolingApiUrl) {
    if (logger.isDebugEnabled()) {
      logger.debug("Waiting for Tooling API URL to be operational...");
    }
    with()
        .timeout(configuration.getStartTimeout() == 0 ? Duration.FOREVER
            : new Duration(configuration.getStartTimeout(), MILLISECONDS))
        .and().with().pollInterval(configuration.getStartPollInterval(), MILLISECONDS).and().with()
        .pollDelay(configuration.getStartPollDelay(), MILLISECONDS).await("Waiting for Remote Tooling Service to be operational")
        .until(() -> {
          final URLConnection urlConnection = toolingApiUrl.openConnection();
          urlConnection.setConnectTimeout(200);
          urlConnection.setReadTimeout(200);
          try {
            urlConnection.connect();
            return true;
          } catch (Exception e) {
            return toolingApiUrl.getProtocol().equals(HTTPS_PROTOCOL) && (e instanceof SSLHandshakeException);
          }
        });
  }

  public boolean isRunning() {
    return muleProcessController.isRunning();
  }

  public void stop() {
    try {
      if (muleProcessController != null && muleProcessController.isRunning()) {
        muleProcessController.stop();
      }
      initialized.getAndSet(false);
    } catch (Exception e) {
      logger.warn("Couldn't stop Mule Runtime", e);
    }
  }

  private URL getToolingApiUrl(String protocol, int restAgentPort) throws MalformedURLException, UnknownHostException {
    return new URL(protocol + "://" + getLocalHost().getHostAddress() + ":" + restAgentPort + MULE_TOOLING_RESOURCE);
  }

}
