/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static org.junit.rules.ExpectedException.none;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.metadata.DataType.STRING;
import static org.mule.runtime.app.declaration.api.fluent.ElementDeclarer.newParameterGroup;
import static org.mule.runtime.app.declaration.api.fluent.ParameterSimpleValue.plain;
import static org.mule.runtime.extension.api.values.ValueBuilder.newValue;

import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionExecutionException;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.value.Value;
import org.mule.runtime.app.declaration.api.ComponentElementDeclaration;
import org.mule.runtime.app.declaration.api.fluent.ElementDeclarer;
import org.mule.tooling.client.internal.session.filter.FieldValueFilter;
import org.mule.tooling.client.internal.session.filter.exception.InvalidLevelValueException;
import org.mule.tooling.client.internal.session.filter.exception.UnknownLevelValueException;

import java.util.HashMap;
import java.util.Set;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class FieldValueFilterTestCase {

  private ExpressionLanguage mockedExpressionLanguage;
  private FieldValueFilter filter;

  @Rule
  public ExpectedException expectedException = none();

  @Before
  public void setUp() {
    mockedExpressionLanguage = mock(ExpressionLanguage.class);
    when(mockedExpressionLanguage.evaluate(anyString(), any(DataType.class), any(BindingContext.class)))
        .thenAnswer(i -> new TypedValue<String>(i.getArgument(0), STRING));
    filter = new FieldValueFilter(mockedExpressionLanguage);
  }

  @Test
  public void filterWithNoConfiguredDeclaration() {
    FieldValueProviderModel fieldValueProviderModel = mockedFieldValueProviderModel("providerName", "targetSelector");
    ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getFieldValueProviderModels()).thenReturn(singletonList(fieldValueProviderModel));
    ComponentElementDeclaration<?> declaration =
        ElementDeclarer.forExtension("SomeExtension").newOperation("operation").getDeclaration();
    Set<Value> values = singleton(
                                  newValue("level0")
                                      .withChild(newValue("level1")
                                          .withChild(newValue("level2")))
                                      .build());

    Set<Value> result = filter.filter(fieldValueProviderModel, parameterModel, declaration, values);
    assertThat(result, hasSize(1));
    assertThat(result.iterator().next().getId(), is("level0"));
  }

  @Test
  public void fieldNotStringFails() {
    TypedValue<?> returnValue = TypedValue.of(new HashMap<>());
    doReturn(returnValue).when(mockedExpressionLanguage).evaluate(anyString(), any(DataType.class), any(BindingContext.class));

    final String parameterName = "parameterName";
    FieldValueProviderModel fieldValueProviderModel = mockedFieldValueProviderModel(parameterName, "level0");
    FieldValueProviderModel otherFieldValueProviderModel = mockedFieldValueProviderModel(parameterName, "level1");
    when(otherFieldValueProviderModel.getPartOrder()).thenReturn(2);
    ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getFieldValueProviderModels()).thenReturn(asList(fieldValueProviderModel, otherFieldValueProviderModel));
    ComponentElementDeclaration<?> declaration =
        ElementDeclarer.forExtension("SomeExtension")
            .newOperation("operation").withParameterGroup(
                                                          newParameterGroup()
                                                              .withParameter(parameterName, plain("parameterValue"))
                                                              .getDeclaration())
            .getDeclaration();
    Set<Value> values = singleton(
                                  newValue("level0")
                                      .withChild(newValue("level1")
                                          .withChild(newValue("level2")))
                                      .build());

    expectedException.expect(UnknownLevelValueException.class);
    filter.filter(otherFieldValueProviderModel, parameterModel, declaration, values);
  }

  @Test
  public void expressionEvaluationFails() {
    doThrow(new ExpressionExecutionException(createStaticMessage("wrongExpression"))).when(mockedExpressionLanguage)
        .evaluate(anyString(), any(DataType.class), any(BindingContext.class));

    final String parameterName = "parameterName";
    FieldValueProviderModel fieldValueProviderModel = mockedFieldValueProviderModel(parameterName, "level0");
    FieldValueProviderModel otherFieldValueProviderModel = mockedFieldValueProviderModel(parameterName, "level1");
    when(otherFieldValueProviderModel.getPartOrder()).thenReturn(2);
    ParameterModel parameterModel = mock(ParameterModel.class);
    when(parameterModel.getFieldValueProviderModels()).thenReturn(asList(fieldValueProviderModel, otherFieldValueProviderModel));
    ComponentElementDeclaration<?> declaration =
        ElementDeclarer.forExtension("SomeExtension")
            .newOperation("operation").withParameterGroup(
                                                          newParameterGroup()
                                                              .withParameter(parameterName, plain("parameterValue"))
                                                              .getDeclaration())
            .getDeclaration();
    Set<Value> values = singleton(
                                  newValue("level0")
                                      .withChild(newValue("level1")
                                          .withChild(newValue("level2")))
                                      .build());

    expectedException.expect(InvalidLevelValueException.class);
    filter.filter(otherFieldValueProviderModel, parameterModel, declaration, values);
  }

  private FieldValueProviderModel mockedFieldValueProviderModel(String providerName, String targetSelector) {
    final FieldValueProviderModel fieldValueProviderModel = mock(FieldValueProviderModel.class);
    when(fieldValueProviderModel.getParameters()).thenReturn(emptyList());
    when(fieldValueProviderModel.getProviderId()).thenReturn(providerName);
    when(fieldValueProviderModel.getProviderName()).thenReturn(providerName + "[]");
    when(fieldValueProviderModel.getTargetSelector()).thenReturn(targetSelector);
    when(fieldValueProviderModel.requiresConfiguration()).thenReturn(false);
    when(fieldValueProviderModel.requiresConnection()).thenReturn(false);
    when(fieldValueProviderModel.getPartOrder()).thenReturn(1);
    when(fieldValueProviderModel.isOpen()).thenReturn(false);
    return fieldValueProviderModel;
  }


}
