/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.session;

import static java.util.Optional.of;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mule.runtime.api.value.ValueResult;
import org.mule.runtime.app.declaration.api.ElementDeclaration;
import org.mule.runtime.core.internal.value.cache.ValueProviderCacheId;
import org.mule.tooling.client.internal.metadata.ToolingCacheIdGenerator;
import org.mule.tooling.client.internal.session.cache.DeclarationValueProviderCache;
import org.mule.tooling.client.internal.session.cache.DefaultDeclarationValueProviderCache;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import org.junit.Before;
import org.junit.Test;

public class DefaultDeclarationValueProviderCacheTestCase {

  private ValueResult valueResult;
  private ValueProviderCacheId cacheId;
  private ElementDeclaration elementDeclaration;
  private ToolingCacheIdGenerator<ElementDeclaration> cacheIdGenerator;

  private Map<String, ValueResult> storage;
  private DeclarationValueProviderCache cache;

  @Before
  public void setUp() {
    valueResult = mock(ValueResult.class);
    cacheId = mock(ValueProviderCacheId.class);
    cacheIdGenerator = mock(ToolingCacheIdGenerator.class);
    elementDeclaration = mock(ElementDeclaration.class);

    storage = new HashMap<>();
    cache = new DefaultDeclarationValueProviderCache(cacheIdGenerator, storage);
  }

  @Test
  public void cacheCallsCallableIfKeyNotFound() {
    final String key = "key";
    when(valueResult.isSuccess()).thenReturn(true);
    when(cacheId.getValue()).thenReturn(key);
    when(cacheIdGenerator.getIdForResolvedValues(eq(elementDeclaration), any())).thenReturn(of(key));

    AtomicBoolean called = new AtomicBoolean(false);

    ValueResult result = cache.getValues(elementDeclaration, "param",
                                         () -> {
                                           called.set(true);
                                           return valueResult;
                                         },
                                         false);

    assertThat(result, equalTo(valueResult));
    assertThat(storage.entrySet(), hasSize(1));
    assertThat(storage.get(key), equalTo(valueResult));
    assertThat(called.get(), is(true));
  }

  @Test
  public void callableNotCalledIfKeyIsPresent() {
    final String key = "key";
    storage.put(key, valueResult);

    when(cacheId.getValue()).thenReturn(key);
    when(cacheIdGenerator.getIdForResolvedValues(eq(elementDeclaration), any())).thenReturn(of(key));

    AtomicBoolean called = new AtomicBoolean(false);

    ValueResult result = cache.getValues(elementDeclaration,
                                         "param",
                                         () -> {
                                           called.set(true);
                                           return valueResult;
                                         },
                                         false);

    assertThat(result, equalTo(valueResult));
    assertThat(storage.entrySet(), hasSize(1));
    assertThat(storage.get(key), equalTo(valueResult));
    assertThat(called.get(), is(false));
  }

  @Test
  public void callableNotCalledIfKeyStored() {
    final String key = "key";
    when(valueResult.isSuccess()).thenReturn(true);
    when(cacheId.getValue()).thenReturn(key);
    when(cacheIdGenerator.getIdForResolvedValues(eq(elementDeclaration), any())).thenReturn(of(key));

    AtomicInteger timesCalled = new AtomicInteger();

    ValueResult result1 = cache.getValues(elementDeclaration,
                                          "param",
                                          () -> {
                                            timesCalled.incrementAndGet();
                                            return valueResult;
                                          },
                                          false);

    ValueResult result2 = cache.getValues(elementDeclaration,
                                          "param",
                                          () -> {
                                            timesCalled.incrementAndGet();
                                            return valueResult;
                                          },
                                          false);


    assertThat(result1, equalTo(valueResult));
    assertThat(result2, equalTo(valueResult));
    assertThat(storage.entrySet(), hasSize(1));
    assertThat(storage.get(key), equalTo(valueResult));
    assertThat(timesCalled.get(), is(1));
  }

  @Test
  public void exceptionWhenGettingValueFromCache() {
    when(valueResult.isSuccess()).thenReturn(true);

    final String key = "key";

    when(cacheId.getValue()).thenReturn(key);
    when(cacheIdGenerator.getIdForResolvedValues(eq(elementDeclaration), any())).thenReturn(of(key));

    AtomicInteger timesCalled = new AtomicInteger();

    Map<String, ValueResult> mockStorage = mock(Map.class);
    when(mockStorage.containsKey(key)).thenReturn(true);
    when(mockStorage.get(key)).thenThrow(new RuntimeException("Expected exception"));

    DeclarationValueProviderCache cache = new DefaultDeclarationValueProviderCache(cacheIdGenerator, mockStorage);

    cache.getValues(elementDeclaration,
                    "param",
                    () -> {
                      timesCalled.incrementAndGet();
                      return valueResult;
                    },
                    false);
    assertThat(timesCalled.get(), is(1));

    verify(mockStorage).containsKey(key);
    verify(mockStorage).get(key);
    verify(mockStorage).put(eq(key), any());
  }

  @Test
  public void nullValue() {
    when(valueResult.isSuccess()).thenReturn(true);

    final String key = "key";

    when(cacheId.getValue()).thenReturn(key);
    when(cacheIdGenerator.getIdForResolvedValues(eq(elementDeclaration), any())).thenReturn(of(key));

    AtomicInteger timesCalled = new AtomicInteger();

    Map<String, ValueResult> mockStorage = mock(Map.class);
    when(mockStorage.containsKey(key)).thenReturn(true);
    when(mockStorage.get(key)).thenReturn(null);

    DeclarationValueProviderCache cache = new DefaultDeclarationValueProviderCache(cacheIdGenerator, mockStorage);

    cache.getValues(elementDeclaration,
                    "param",
                    () -> {
                      timesCalled.incrementAndGet();
                      return valueResult;
                    },
                    false);
    assertThat(timesCalled.get(), is(1));

    verify(mockStorage).containsKey(key);
    verify(mockStorage).get(key);
    verify(mockStorage).put(eq(key), any());
  }

  @Test
  public void valuesNotStoredIfFailure() {
    when(valueResult.isSuccess()).thenReturn(false);

    final String key = "key";

    when(cacheId.getValue()).thenReturn(key);
    when(cacheIdGenerator.getIdForResolvedValues(eq(elementDeclaration), any())).thenReturn(of(key));

    AtomicInteger timesCalled = new AtomicInteger();

    cache.getValues(elementDeclaration,
                    "param",
                    () -> {
                      timesCalled.incrementAndGet();
                      return valueResult;
                    },
                    false);
    cache.getValues(elementDeclaration,
                    "param",
                    () -> {
                      timesCalled.incrementAndGet();
                      return valueResult;
                    },
                    false);
    assertThat(timesCalled.get(), is(2));
  }

}
