/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.serialization;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.IsEqual.equalTo;
import org.mule.tooling.client.api.datasense.storage.MetadataCacheStorageSerializer;
import org.mule.tooling.client.internal.cache.DefaultCacheStorageSerializer;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;

import org.junit.Test;

public class DefaultMetadataCacheStorageSerializerTestCase {

  private MetadataCacheStorageSerializer serializer;
  private TestingObject testingObject;

  public DefaultMetadataCacheStorageSerializerTestCase() {
    this.serializer = new DefaultCacheStorageSerializer();
    Map<String, TestingObject> level2 = new HashMap<>();
    level2.put("level2a", new TestingObject("level2a", new HashMap<>()));
    level2.put("level2b", new TestingObject("level2b", new HashMap<>()));
    Map<String, TestingObject> level1 = new HashMap<>();
    level2.put("level1", new TestingObject("level2", level2));
    this.testingObject = new TestingObject("testingObject", level1);
  }

  private byte[] serialize(Object object) throws Exception {
    try (OutputStream byteArrayOutputStream = new ByteArrayOutputStream()) {
      serializer.serialize(byteArrayOutputStream, object);
      return ((ByteArrayOutputStream) byteArrayOutputStream).toByteArray();
    }
  }

  private <T> T deserialize(byte[] bytes) throws Exception {
    try (InputStream objectInputStream = new ByteArrayInputStream(bytes)) {
      return serializer.deserialize(objectInputStream);
    }
  }

  @Test
  public void serializeAndDeserialize() throws Exception {
    byte[] serializedObject = serialize(testingObject);
    TestingObject deserialized = deserialize(serializedObject);
    assertThat(deserialized, equalTo(testingObject));
  }

  @Test
  public void serializeAndDeserializeEncodingInBase64() throws Exception {
    byte[] serializedObject = Base64.getEncoder().encode(serialize(testingObject));
    TestingObject deserialized = deserialize(Base64.getDecoder().decode(serializedObject));
    assertThat(deserialized, equalTo(testingObject));
  }

  @Test
  public void serializeAndDeserializeBuildingString() throws Exception {
    String serializedObject = new String(serialize(testingObject));
    TestingObject deserialized = deserialize(serializedObject.getBytes());
    assertThat(deserialized, equalTo(testingObject));
  }

  private class TestingObject {

    private String simpleField;
    private Map<String, TestingObject> complexField;

    public TestingObject(String simpleField, Map<String, TestingObject> complexField) {
      this.simpleField = simpleField;
      this.complexField = complexField;
    }

    @Override
    public int hashCode() {
      int result = simpleField.hashCode();
      result = 31 * result + complexField.hashCode();
      return result;
    }

    @Override
    public boolean equals(Object obj) {
      if (obj == this) {
        return true;
      }
      if (!(obj instanceof TestingObject)) {
        return false;
      }
      TestingObject other = (TestingObject) obj;
      return this.simpleField.equals(other.simpleField) && this.complexField.equals(other.complexField);
    }
  }

}
