/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.metadata;

import static java.util.Optional.of;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.sameInstance;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.ast.api.ComponentAst;

import java.util.HashMap;
import java.util.Map;

import com.google.common.collect.ImmutableList;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.junit.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class MetadataCachePopulatorTestCase extends AbstractMetadataTestCase {

  private static final String KEY = "testKey";

  private Map<String, MetadataType> metadataTypesCache;
  private ToolingCacheIdGenerator<ComponentAst> cacheIdGenerator;
  private ComponentAst componentAst;

  @Before
  public void prepare() {
    metadataTypesCache = new HashMap<>();
    cacheIdGenerator = mock(ToolingCacheIdGenerator.class);
    componentAst = mock(ComponentAst.class);
  }

  // OperationModel
  @Test
  public void outputMetadataShouldBeAddedToCacheForOperationModel() {
    OperationModel operationModel =
        getOperationModel(ImmutableList.of(staticParameterModel), dynamicOutputModel, staticOutputModel);

    when(cacheIdGenerator.getIdForComponentOutputMetadata(componentAst)).thenReturn(of(KEY));

    new MetadataCachePopulator<>(metadataTypesCache, cacheIdGenerator, componentAst, operationModel).populateCache();

    assertThat(metadataTypesCache.isEmpty(), is(false));
    assertThat(metadataTypesCache.size(), is(1));
    assertThat(metadataTypesCache.get(KEY), sameInstance(stringType));
  }

  @Test
  public void outputAttributesShouldBeAddedToCacheForOperationModel() {
    OperationModel operationModel =
        getOperationModel(ImmutableList.of(staticParameterModel), staticOutputModel, dynamicOutputModel);

    when(cacheIdGenerator.getIdForComponentAttributesMetadata(componentAst)).thenReturn(of(KEY));

    new MetadataCachePopulator<>(metadataTypesCache, cacheIdGenerator, componentAst, operationModel).populateCache();

    assertThat(metadataTypesCache.isEmpty(), is(false));
    assertThat(metadataTypesCache.size(), is(1));
    assertThat(metadataTypesCache.get(KEY), sameInstance(stringType));
  }

  @Test
  public void outputAttributesShouldNotBeOverwrittenToCacheForOperationModel() {
    OperationModel operationModel =
        getOperationModel(ImmutableList.of(staticParameterModel), staticOutputModel, dynamicOutputModel);

    when(cacheIdGenerator.getIdForComponentAttributesMetadata(componentAst)).thenReturn(of(KEY));

    metadataTypesCache.put(KEY, objectType);

    new MetadataCachePopulator<>(metadataTypesCache, cacheIdGenerator, componentAst, operationModel).populateCache();

    assertThat(metadataTypesCache.isEmpty(), is(false));
    assertThat(metadataTypesCache.size(), is(1));
    assertThat(metadataTypesCache.get(KEY), sameInstance(objectType));
  }

  @Test
  public void parameterShouldBeAddedToCacheForOperationModel() {
    OperationModel operationModel =
        getOperationModel(ImmutableList.of(staticParameterModel, dynamicParameterModel), staticOutputModel, staticOutputModel);

    when(cacheIdGenerator.getIdForComponentInputMetadata(componentAst, dynamicParameterModel.getName()))
        .thenReturn(of(KEY));

    new MetadataCachePopulator<>(metadataTypesCache, cacheIdGenerator, componentAst, operationModel).populateCache();

    assertThat(metadataTypesCache.isEmpty(), is(false));
    assertThat(metadataTypesCache.size(), is(1));
    assertThat(metadataTypesCache.get(KEY), sameInstance(stringType));
  }

  // SourceModel
  @Test
  public void outputMetadataShouldBeAddedToCacheForSourceModel() {
    SourceModel sourceModel =
        getSourceModel(ImmutableList.of(staticParameterModel), dynamicOutputModel, staticOutputModel);

    when(cacheIdGenerator.getIdForComponentOutputMetadata(componentAst)).thenReturn(of(KEY));

    new MetadataCachePopulator<>(metadataTypesCache, cacheIdGenerator, componentAst, sourceModel).populateCache();

    assertThat(metadataTypesCache.isEmpty(), is(false));
    assertThat(metadataTypesCache.size(), is(1));
    assertThat(metadataTypesCache.get(KEY), sameInstance(stringType));
  }

  @Test
  public void outputAttributesShouldBeAddedToCacheForSourceModel() {
    SourceModel operationModel =
        getSourceModel(ImmutableList.of(staticParameterModel), staticOutputModel, dynamicOutputModel);

    when(cacheIdGenerator.getIdForComponentAttributesMetadata(componentAst)).thenReturn(of(KEY));

    new MetadataCachePopulator<>(metadataTypesCache, cacheIdGenerator, componentAst, operationModel).populateCache();

    assertThat(metadataTypesCache.isEmpty(), is(false));
    assertThat(metadataTypesCache.size(), is(1));
    assertThat(metadataTypesCache.get(KEY), sameInstance(stringType));
  }

  @Test
  public void outputAttributesShouldNotBeOverwrittenToCacheForSourceModel() {
    SourceModel sourceModel =
        getSourceModel(ImmutableList.of(staticParameterModel), staticOutputModel, dynamicOutputModel);

    when(cacheIdGenerator.getIdForComponentAttributesMetadata(componentAst)).thenReturn(of(KEY));

    metadataTypesCache.put(KEY, objectType);

    new MetadataCachePopulator<>(metadataTypesCache, cacheIdGenerator, componentAst, sourceModel).populateCache();

    assertThat(metadataTypesCache.isEmpty(), is(false));
    assertThat(metadataTypesCache.size(), is(1));
    assertThat(metadataTypesCache.get(KEY), sameInstance(objectType));
  }

  @Test
  public void parameterShouldBeAddedToCacheForSourceModel() {
    SourceModel operationModel =
        getSourceModel(ImmutableList.of(staticParameterModel, dynamicParameterModel), staticOutputModel, staticOutputModel);

    when(cacheIdGenerator.getIdForComponentInputMetadata(componentAst, dynamicParameterModel.getName()))
        .thenReturn(of(KEY));

    new MetadataCachePopulator<>(metadataTypesCache, cacheIdGenerator, componentAst, operationModel).populateCache();

    assertThat(metadataTypesCache.isEmpty(), is(false));
    assertThat(metadataTypesCache.size(), is(1));
    assertThat(metadataTypesCache.get(KEY), sameInstance(stringType));
  }
}
