/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.hamcrest;

import static java.util.Optional.empty;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.mule.tooling.client.internal.hamcrest.HamcrestUtils.validateThat;

import org.mule.runtime.extension.api.property.TypeResolversInformationModelProperty;
import org.mule.tooling.client.api.extension.model.metadata.TypeResolversInformationModel;

import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;

public class TypeResolverInformationModelMatcher extends TypeSafeDiagnosingMatcher<TypeResolversInformationModel> {

  public static Matcher<TypeResolversInformationModel> from(TypeResolversInformationModelProperty runtimeModelProperty) {
    if (runtimeModelProperty == null) {
      return nullValue(TypeResolversInformationModel.class);
    }
    return new TypeResolverInformationModelMatcher(runtimeModelProperty);
  }

  private final TypeResolversInformationModelProperty runtimeModelProperty;

  private TypeResolverInformationModelMatcher(TypeResolversInformationModelProperty runtimeModelProperty) {
    this.runtimeModelProperty = runtimeModelProperty;
  }

  @Override
  protected boolean matchesSafely(TypeResolversInformationModel item, Description description) {
    return validateThat("category", item.getCategoryName(), equalTo(runtimeModelProperty.getCategoryName()), description) &&
        runtimeModelProperty
            .getOutputResolver()
            .map(ror -> item
                .getOutputResolver()
                .map(or -> validateThat("outputResolver", or, ResolverInformationMatcher.from(ror), description))
                .orElse(false))
            .orElseGet(() -> validateThat("outputResolver", item.getOutputResolver(), equalTo(empty()), description))
        &&

        runtimeModelProperty
            .getAttributesResolver()
            .map(ror -> item
                .getAttributesResolver()
                .map(or -> validateThat("outputAttributes", or, ResolverInformationMatcher.from(ror), description))
                .orElse(false))
            .orElseGet(() -> validateThat("outputAttributes", item.getAttributesResolver(), equalTo(empty()), description))
        &&

        runtimeModelProperty
            .getKeysResolver()
            .map(ror -> item
                .getKeysResolver()
                .map(or -> validateThat("keys", or, ResolverInformationMatcher.from(ror), description))
                .orElse(false))
            .orElseGet(() -> validateThat("keys", item.getKeysResolver(), equalTo(empty()), description))
        &&

        item
            .getParameterResolvers()
            .entrySet()
            .stream()
            .allMatch(e -> runtimeModelProperty
                .getParameterResolver(e.getKey())
                .map(rpr -> validateThat("parameter: " + e.getKey(), e.getValue(), ResolverInformationMatcher.from(rpr),
                                         description))
                .orElseGet(() -> validateThat("parameter: " + e.getKey(), e.getValue(), nullValue(), description)));

  }

  @Override
  public void describeTo(Description description) {
    description.appendText("TypeResolverInformation: ").appendValue(runtimeModelProperty);
  }
}
