/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.hamcrest;

import static java.util.Optional.empty;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.collection.IsIterableContainingInOrder.contains;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.hamcrest.core.IsNull.nullValue;
import static org.mule.tooling.client.internal.hamcrest.HamcrestUtils.validateThat;

import org.mule.tooling.client.api.extension.model.nested.NestedComponentModel;

import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;

public class NestedComponentMatcher<T extends NestedComponentModel> extends TypeSafeDiagnosingMatcher<T> {

  public static <T extends NestedComponentModel, R extends org.mule.runtime.api.meta.model.nested.NestedComponentModel> Matcher<T> from(R runtimeNestedComponentModel) {
    return new NestedComponentMatcher<>(runtimeNestedComponentModel);
  }

  protected final org.mule.runtime.api.meta.model.nested.NestedComponentModel runtimeNestedComponentModel;

  protected NestedComponentMatcher(org.mule.runtime.api.meta.model.nested.NestedComponentModel runtimeNestedComponentModel) {
    this.runtimeNestedComponentModel = runtimeNestedComponentModel;
  }


  @Override
  protected boolean matchesSafely(NestedComponentModel nestedComponentModel, Description description) {
    return validateThat("isRequired", nestedComponentModel.isRequired(), equalTo(runtimeNestedComponentModel.isRequired()),
                        description)
        &&
        validateThat("name", nestedComponentModel.getName(), equalTo(runtimeNestedComponentModel.getName()), description) &&
        validateThat("description", nestedComponentModel.getDescription(), equalTo(runtimeNestedComponentModel.getDescription()),
                     description)
        &&

        runtimeNestedComponentModel
            .getDisplayModel()
            .map(rdp -> validateThat("displayModel", nestedComponentModel.getDisplayModel(),
                                     DisplayModelMatcher.from(rdp),
                                     description))
            .orElseGet(() -> validateThat("displayModel", nestedComponentModel.getDisplayModel(), nullValue(), description))
        &&

        (runtimeNestedComponentModel.getAllowedStereotypes().isEmpty()
            ? validateThat("allowedStereotypes", nestedComponentModel.getAllowedStereotypes(), hasSize(0), description)
            : validateThat("allowedStereotypes", nestedComponentModel.getAllowedStereotypes(),
                           contains(StereotypeMatcher.sFrom(runtimeNestedComponentModel.getAllowedStereotypes())),
                           description))
        &&

        (!nestedComponentModel.getParameterGroupModels().isEnabled() ||
            runtimeNestedComponentModel.getParameterGroupModels().isEmpty()
                ? validateThat("parameterGroupModels", nestedComponentModel.getParameterGroupModels().get(), hasSize(0),
                               description)
                : validateThat("parameterGroupModels",
                               nestedComponentModel.getParameterGroupModels().get(),
                               contains(ParameterGroupMatcher.sFrom(runtimeNestedComponentModel.getParameterGroupModels())),
                               description))

        &&

        (!nestedComponentModel.getDeprecationModel().isEnabled() ||
            runtimeNestedComponentModel
                .getDeprecationModel()
                .map(rdp -> validateThat("deprecationModel", nestedComponentModel.getDeprecationModel().get(),
                                         DeprecationModelMatcher.from(rdp), description))
                .orElseGet(() -> validateThat("deprecationModel", nestedComponentModel.getDeprecationModel().get(), nullValue(),
                                              description)))
        &&

        (!nestedComponentModel.getErrorModels().isEnabled() ||
            runtimeNestedComponentModel.getErrorModels().isEmpty()
                ? validateThat("errorModels", nestedComponentModel.getErrorModels().get(), hasSize(0), description)
                : validateThat(
                               "errorModels",
                               nestedComponentModel.getErrorModels().get(),
                               contains(ErrorModelMatcher.sFrom(runtimeNestedComponentModel.getErrorModels())),
                               description))

        &&

        (!nestedComponentModel.getMinOccurs().isEnabled() ||
            validateThat("minOccurs", nestedComponentModel.getMinOccurs().get(),
                         equalTo(runtimeNestedComponentModel.getMinOccurs()), description))
        &&

        (!nestedComponentModel.getMaxOccurs().isEnabled() ||
            runtimeNestedComponentModel
                .getMaxOccurs()
                .map(rmo -> nestedComponentModel.getMaxOccurs().get()
                    .map(mo -> validateThat("maxOccurs", mo, equalTo(rmo), description))
                    .orElse(false))
                .orElseGet(() -> validateThat("maxOccurs", nestedComponentModel.getMaxOccurs().get(), equalTo(empty()),
                                              description)))
        &&

        (!nestedComponentModel.getNestedComponents().isEnabled() ||
            runtimeNestedComponentModel.getNestedComponents().isEmpty()
                ? validateThat("nestedComponents", nestedComponentModel.getNestedComponents().get(), hasSize(0), description)
                : validateThat("nestedComponents",
                               nestedComponentModel.getNestedComponents().get(),
                               contains(NestableElementMatcher.sFrom(runtimeNestedComponentModel.getNestedComponents())),
                               description))
        &&

        (!nestedComponentModel.getStereotype().isEnabled() ||
            (runtimeNestedComponentModel.getStereotype() == null
                ? validateThat("stereotype", nestedComponentModel.getStereotype().get(), nullValue(), description)
                : validateThat("stereotype",
                               nestedComponentModel.getStereotype().get(),
                               StereotypeMatcher.from(runtimeNestedComponentModel.getStereotype()),
                               description)))

        && validateThat("semanticTerms", nestedComponentModel.getSemanticTerms(),
                        equalTo(runtimeNestedComponentModel.getSemanticTerms()), description);
  }

  @Override
  public void describeTo(Description description) {
    description.appendText("NestedComponentModel: ").appendValue(runtimeNestedComponentModel);
  }
}
