/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.hamcrest;

import static java.util.Optional.empty;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.mule.tooling.client.internal.hamcrest.HamcrestUtils.validateThat;

import org.mule.tooling.client.api.extension.model.ExternalLibraryModel;
import java.util.Collection;

import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;

public class ExternalLibraryModelMatcher extends TypeSafeDiagnosingMatcher<ExternalLibraryModel> {

  public static Matcher<ExternalLibraryModel> from(org.mule.runtime.api.meta.model.ExternalLibraryModel runtimeExternalLibrary) {
    if (runtimeExternalLibrary == null) {
      return nullValue(ExternalLibraryModel.class);
    }
    return new ExternalLibraryModelMatcher(runtimeExternalLibrary);
  }

  public static Matcher<ExternalLibraryModel>[] sFrom(Collection<org.mule.runtime.api.meta.model.ExternalLibraryModel> runtimeExternalLibraryModels) {
    return runtimeExternalLibraryModels.stream().map(ExternalLibraryModelMatcher::from)
        .toArray(ExternalLibraryModelMatcher[]::new);
  }

  private final org.mule.runtime.api.meta.model.ExternalLibraryModel runtimeExternalLibrary;

  private ExternalLibraryModelMatcher(org.mule.runtime.api.meta.model.ExternalLibraryModel runtimeExternalLibrary) {
    this.runtimeExternalLibrary = runtimeExternalLibrary;
  }

  @Override
  protected boolean matchesSafely(ExternalLibraryModel item, Description mismatchDescription) {
    return validateThat("name", item.getName(), equalTo(runtimeExternalLibrary.getName()), mismatchDescription) &&
        validateThat("description", item.getDescription(), equalTo(runtimeExternalLibrary.getDescription()), mismatchDescription)
        &&

        runtimeExternalLibrary
            .getRequiredClassName()
            .map(rrcn -> item.getRequiredClassName()
                .map(rcn -> validateThat("requiredClassName", rcn, equalTo(rrcn), mismatchDescription))
                .orElse(false))
            .orElseGet(() -> validateThat("requiredClassName", item.getRequiredClassName(), equalTo(empty()),
                                          mismatchDescription))
        &&

        runtimeExternalLibrary
            .getRegexMatcher()
            .map(rrm -> item.getRegexMatcher()
                .map(rm -> validateThat("regexMatcher", rm, equalTo(rrm), mismatchDescription))
                .orElse(false))
            .orElseGet(() -> validateThat("regexMatcher", item.getRegexMatcher(), equalTo(empty()), mismatchDescription))
        &&

        validateThat("type", item.getType(), ExternalLibraryTypeMatcher.from(runtimeExternalLibrary.getType()),
                     mismatchDescription)
        &&

        runtimeExternalLibrary
            .getSuggestedCoordinates()
            .map(rsc -> item.getSuggestedCoordinates()
                .map(sc -> validateThat("suggestedCoordinates", sc, equalTo(rsc), mismatchDescription))
                .orElse(false))
            .orElseGet(() -> validateThat("suggestedCoordinates", item.getSuggestedCoordinates(), equalTo(empty()),
                                          mismatchDescription))
        &&

        validateThat("isOptional", item.isOptional(), equalTo(runtimeExternalLibrary.isOptional()), mismatchDescription);
  }

  @Override
  public void describeTo(Description description) {
    description.appendText("ExternalLibraryModel: ").appendValue(runtimeExternalLibrary);
  }
}
