/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.dataweave;

import org.mule.runtime.api.metadata.MediaType;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewRequest;
import org.mule.tooling.event.model.DataTypeModel;
import org.mule.tooling.event.model.EventModel;
import org.mule.tooling.event.model.MessageModel;
import org.mule.tooling.event.model.TypedValueModel;
import org.mule.weave.v2.runtime.utils.WeaveRequirementsChecker;

import java.nio.charset.Charset;
import java.util.Map;

public class DataWeaveRunnerProvider {

  private static final String DW_MEDIA_TYPE = "application/dw";

  private IDataWeaveRunner localRunner;
  private IDataWeaveRunner remoteRunner;

  public DataWeaveRunnerProvider(IDataWeaveRunner localRunner, IDataWeaveRunner remoteRunner) {
    this.localRunner = localRunner;
    this.remoteRunner = remoteRunner;
  }

  public IDataWeaveRunner getRunner(DataWeavePreviewRequest parameters) {
    WeaveRequirementsChecker requirementsChecker = new WeaveRequirementsChecker();
    if (requirementsChecker.requiresJava(parameters.getScript()) || parametersRequiresJava(parameters.getEvent())) {
      return getRemoteRunner();
    } else {
      return getLocalRunner();
    }
  }

  private boolean parametersRequiresJava(EventModel model) {
    if (model != null) {
      if (messageRequiresJava(model.getMessage())) {
        return true;
      }
      if (variablesRequiresJava(model)) {
        return true;
      }
    }

    return false;
  }

  private boolean variablesRequiresJava(EventModel model) {
    Map<String, TypedValueModel> variables = model.getVariables();
    if (variables != null && !variables.isEmpty()) {
      boolean anyRequiresJava = variables.entrySet().stream().anyMatch(entry -> isJavaBased(entry.getValue()));
      if (anyRequiresJava) {
        return true;
      }
    }
    return false;
  }

  private boolean messageRequiresJava(MessageModel message) {
    if (message != null) {
      TypedValueModel payload = message.getPayload();
      if (payload != null && isJavaBased(payload)) {
        return true;
      }
      TypedValueModel attributes = message.getAttributes();
      if (attributes != null && isJavaBased(attributes)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Determines if the current typedValueModel is a DW script that outputs to a Java object.
   * 
   * @param typedValueModel
   * @return <code>true</code> if media type is DW and output requires Java
   */
  private boolean isJavaBased(final TypedValueModel typedValueModel) {
    final DataTypeModel dataType = typedValueModel.getDataType();
    final String mediaTypeModel = dataType.getMediaType();
    final MediaType mediaType = MediaType.parse(mediaTypeModel);

    final Charset charset = mediaType.getCharset().orElseGet(Charset::defaultCharset);

    return DW_MEDIA_TYPE.equals(mediaType.getPrimaryType())
        && new WeaveRequirementsChecker().requiresJava(new String(typedValueModel.getContent(), charset));
  }

  public IDataWeaveRunner getLocalRunner() {
    return localRunner;
  }

  public IDataWeaveRunner getRemoteRunner() {
    return remoteRunner;
  }
}
