/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static java.util.Objects.requireNonNull;
import static org.apache.commons.io.IOUtils.copy;
import static org.mule.runtime.core.api.util.IOUtils.closeQuietly;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.client.api.exception.ToolingException;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.net.URLConnection;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Handles an {@link InputStream} with the content of the zipped application from local file system or a remote server.
 *
 * @since 4.0
 */
public class InputStreamApplicationDeployer implements ApplicationDeployer {

  private static final int CONNECT_TIMEOUT = 5000;
  private static final int READ_TIMEOUT = 5000;

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private URL applicationUrl;
  private RuntimeToolingService runtimeToolingService;

  public InputStreamApplicationDeployer(URL applicationUrl, RuntimeToolingService runtimeToolingService) {
    requireNonNull(applicationUrl, "applicationUrl cannot be null");
    requireNonNull(runtimeToolingService, "runtimeToolingService cannot be null");

    this.applicationUrl = applicationUrl;
    this.runtimeToolingService = runtimeToolingService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String deploy() {
    if (logger.isDebugEnabled()) {
      logger.debug("Deploying application from URL: {}", applicationUrl);
    }

    // TODO (gfernandes) MULE-11149
    byte[] content = readContentFromUrl(applicationUrl);

    return runtimeToolingService.deployApplication(new ByteArrayInputStream(content));
  }

  public static byte[] readContentFromUrl(URL applicationUrl) {
    byte[] content;
    InputStream inputStream = null;
    OutputStream outputStream = null;
    try {
      URLConnection urlConnection = applicationUrl.openConnection();
      urlConnection.setConnectTimeout(CONNECT_TIMEOUT);
      urlConnection.setReadTimeout(READ_TIMEOUT);
      inputStream = new BufferedInputStream(urlConnection.getInputStream());
      ByteArrayOutputStream bufferOutputStream = new ByteArrayOutputStream();
      copy(inputStream, bufferOutputStream);
      content = bufferOutputStream.toByteArray();
    } catch (Exception e) {
      throw new ToolingException("Error while reading application content from URL: " + applicationUrl, e);
    } finally {
      closeQuietly(outputStream);
      closeQuietly(inputStream);
    }
    return content;
  }

}
