/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkState;
import org.mule.tooling.agent.RuntimeToolingService;

import java.util.ServiceLoader;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Uses SPI to get the {@link RuntimeToolingService} implementation. Allowing to have different implementations for the
 * service.
 *
 * @since 4.0
 */
class SpiRuntimeToolingServiceProvider implements Supplier<RuntimeToolingService> {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private transient volatile boolean initialized;
  private RuntimeToolingService runtimeToolingService;

  /**
   * Creates an instance of the controller.
   */
  SpiRuntimeToolingServiceProvider() {}

  /**
   * {@inheritDoc}
   */
  public RuntimeToolingService get() {
    if (!this.initialized) {
      synchronized (this) {
        if (!this.initialized) {
          this.runtimeToolingService = lookUpToolingService();
          this.initialized = true;
          return runtimeToolingService;
        }
      }
    }
    return this.runtimeToolingService;
  }

  private RuntimeToolingService lookUpToolingService() {
    if (logger.isDebugEnabled()) {
      logger.debug("Using SPI mechanism to look for the implementation of {} to be used by the client",
                   RuntimeToolingService.class.getName());
    }

    ServiceLoader<RuntimeToolingService> serviceLoader =
        ServiceLoader.load(RuntimeToolingService.class, this.getClass().getClassLoader());
    checkState(serviceLoader.iterator().hasNext(), "No service found for: '" + RuntimeToolingService.class.getName() + "'");

    return serviceLoader.iterator().next();
  }

}
