/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Suppliers.memoize;
import org.mule.maven.client.api.MavenClient;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.connectivity.ConnectivityTestingService;
import org.mule.tooling.client.api.datasense.DataSenseService;
import org.mule.tooling.client.api.dataweave.DataWeaveService;
import org.mule.tooling.client.api.metadata.MetadataService;
import org.mule.tooling.client.internal.application.Application;
import org.mule.tooling.client.internal.application.FetchedApplication;
import org.mule.tooling.client.internal.dataweave.DataWeaveRunnerProvider;
import org.mule.tooling.client.internal.dataweave.DefaultDataWeaveService;
import org.mule.tooling.client.internal.dataweave.IDataWeaveRunner;
import org.mule.tooling.client.internal.dataweave.LocalRunner;
import org.mule.tooling.client.internal.dataweave.ModulesAnalyzer;
import org.mule.tooling.client.internal.dataweave.RemoteRunner;

import java.net.URL;
import java.util.function.Supplier;

/**
 * Implementation that allows to do Tooling operations without deploying the application until it is needed.
 * <p/>
 * If an operation requires the application to be deployed it will deploy it first and then delegate the operation to the default
 * implementation of {@link ToolingArtifact}.
 *
 * @since 4.0
 */
public class DefaultToolingArtifact implements ToolingArtifact {

  private Application application;

  private ConnectivityTestingService connectivityTestingService;
  private MetadataService metadataService;
  private DataSenseService dataSenseService;
  private DataWeaveService dataWeaveService;

  /**
   * Creates an instance of the {@link DefaultToolingArtifact} from a fetched applicationId or deploys the application to obtain
   * an identifier in case if null.
   *
   * @param applicationId the identifier of the application already deployed on Mule Runtime. Can be null. If null, the
   *        application will be deployed and identifier obtained from the deployment.
   * @param applicationUrlSupplier {@link URL} to the application location, expanded or a zip.
   * @param runtimeToolingServiceSupplier {@link Supplier} for {@link RuntimeToolingService} in order to access remote operations.
   *        Non null.
   */
  public DefaultToolingArtifact(String applicationId, Supplier<URL> applicationUrlSupplier,
                                Supplier<RuntimeToolingService> runtimeToolingServiceSupplier,
                                MavenClient mavenClient,
                                MuleRuntimeExtensionModelProvider runtimeExtensionModelProvider,
                                ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader) {
    checkNotNull(applicationUrlSupplier, "applicationUrlSupplier cannot be null");
    checkNotNull(runtimeToolingServiceSupplier, "runtimeToolingServiceSupplier cannot be null");

    // Use memoize Supplier from Guava as it has to be passed to different objects and only once needs to be called.
    applicationUrlSupplier = memoize(applicationUrlSupplier::get)::get;
    runtimeToolingServiceSupplier = memoize(runtimeToolingServiceSupplier::get)::get;

    if (applicationId != null) {
      this.application = new FetchedApplication(applicationId, applicationUrlSupplier, runtimeToolingServiceSupplier, mavenClient,
                                                runtimeExtensionModelProvider);
    } else {
      this.application =
          new Application(applicationUrlSupplier, runtimeToolingServiceSupplier, mavenClient, runtimeExtensionModelProvider);
    }

    // Services registration...
    this.connectivityTestingService = new DefaultConnectivityTestingService(() -> application.deploy());
    MetadataProvider metadataProvider = new InternalMetadataProvider(() -> application.deploy());
    this.metadataService = new ToolingMetadataServiceAdapter(metadataProvider);
    this.dataSenseService =
        new DefaultDataSenseService(applicationUrlSupplier, metadataProvider, application,
                                    componentBuildingDefinitionLoader);
    IDataWeaveRunner remoteRunner = new RemoteRunner(() -> application.deploy());
    IDataWeaveRunner localRunner = new LocalRunner();
    this.dataWeaveService =
        new DefaultDataWeaveService(new DataWeaveRunnerProvider(localRunner, remoteRunner), new ModulesAnalyzer());
  }

  /**
   * Creates an instance of the {@link DefaultToolingArtifact}.
   *
   * @param applicationUrlSupplier {@link URL} to the application location, expanded or a zip.
   * @param runtimeToolingServiceSupplier {@link Supplier} for {@link RuntimeToolingService} in order to access remote operations.
   *        Non null.
   * @param mavenClient maven client to use for resolving dependencies
   * @param runtimeExtensionModelProvider provider for {@link org.mule.runtime.api.meta.model.ExtensionModel}s
   * @param componentBuildingDefinitionLoader loader for {@link org.mule.runtime.dsl.api.component.ComponentBuildingDefinition}s
   */
  public DefaultToolingArtifact(Supplier<URL> applicationUrlSupplier,
                                Supplier<RuntimeToolingService> runtimeToolingServiceSupplier,
                                MavenClient mavenClient,
                                MuleRuntimeExtensionModelProvider runtimeExtensionModelProvider,
                                ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader) {
    this(null, applicationUrlSupplier, runtimeToolingServiceSupplier, mavenClient, runtimeExtensionModelProvider,
         componentBuildingDefinitionLoader);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getApplicationId() {
    return application.deploy().getApplicationId();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivityTestingService connectivityTestingService() {
    return this.connectivityTestingService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataService metadataService() {
    return this.metadataService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public org.mule.tooling.client.api.datasense.DataSenseService dataSenseService() {
    return this.dataSenseService;
  }

  @Override
  public DataWeaveService dataweaveService() {
    return dataWeaveService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void dispose() {
    application.dispose();
  }

}
