/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.lang.String.format;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.FileUtils.readFileToString;
import static org.apache.commons.io.FileUtils.toFile;
import static org.mule.runtime.deployment.model.api.application.ApplicationDescriptor.DEFAULT_CONFIGURATION_RESOURCE_LOCATION;
import org.mule.datasense.api.metadataprovider.ApplicationModel;
import org.mule.datasense.api.metadataprovider.DefaultApplicationModel;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.persistence.MuleApplicationModelJsonSerializer;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.deployment.model.api.application.ApplicationDescriptor;
import org.mule.tooling.client.api.exception.ToolingException;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.file.Paths;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

import org.apache.commons.io.IOUtils;

/**
 * Implementation that handles a {@link URL} to local file system referencing to a expanded application content.
 *
 * @since 4.0
 */
public class ApplicationModelFactoryFromExpandedFolder extends BaseApplicationModelFactory {

  /**
   * {@inheritDoc}
   */
  public ApplicationModelFactoryFromExpandedFolder(ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader) {
    super(componentBuildingDefinitionLoader);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<ApplicationModel> createApplicationModel(URL applicationUrl, Set<ExtensionModel> extensionModels) {
    final File file = toFile(applicationUrl);
    return processAppLocation(file, extensionModels)
        .map(applicationModel -> new DefaultApplicationModel(file.getAbsolutePath(), applicationModel, loadTypesData(file)));
  }

  private String loadTypesData(File appLocation) {
    try {
      return readFileToString(new File(new File(appLocation, MULE), APP_TYPES_DATA));
    } catch (IOException e) {
      return null;
    }
  }

  private Optional<Stream<String>> findConfigsFromApplicationDescriptor(File appLocation) throws IOException {
    File muleApplicationJson = new File(appLocation, ApplicationDescriptor.MULE_APPLICATION_JSON_LOCATION);
    if (muleApplicationJson.exists()) {
      try (InputStream stream = new FileInputStream(muleApplicationJson)) {
        MuleApplicationModelJsonSerializer applicationDescriptor = new MuleApplicationModelJsonSerializer();
        MuleApplicationModel applicationModel = applicationDescriptor.deserialize(IOUtils.toString(stream));
        if (!applicationModel.getConfigs().isEmpty()) {
          return of(applicationModel.getConfigs().stream().map(configFileName -> Paths.get(MULE, configFileName).toString()));
        }
      } catch (IOException e) {
        throw new IllegalArgumentException(format("Could not read extension describer on plugin '%s'",
                                                  muleApplicationJson.getAbsolutePath()),
                                           e);
      }
    }
    return of(Stream.of(DEFAULT_CONFIGURATION_RESOURCE_LOCATION));
  }

  private Optional<org.mule.runtime.config.spring.dsl.model.ApplicationModel> readConfigs(Stream<String> configs,
                                                                                          File appLocation,
                                                                                          Set<ExtensionModel> extensionModels) {
    try {
      return of(loadApplicationModel(configs
          .map(configFileName -> {
            try {
              final File configFile = new File(appLocation, configFileName);
              return loadConfigFile(configFileName, new FileInputStream(configFile), extensionModels).get();
            } catch (FileNotFoundException e) {
              throw new ToolingException(e);
            }
          })
          .collect(toList()), extensionModels));
    } catch (Exception e) {
      throw new ToolingException(e);
    }
  }

  private Optional<org.mule.runtime.config.spring.dsl.model.ApplicationModel> processAppLocation(File appLocation,
                                                                                                 Set<ExtensionModel> extensionModels) {
    try {
      return findConfigsFromApplicationDescriptor(appLocation).map(configs -> readConfigs(configs, appLocation, extensionModels))
          .orElse(empty());
    } catch (IOException e) {
      throw new ToolingException(e);
    }
  }

}
