/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.extension.model.parameter;

import static java.util.Optional.ofNullable;
import org.mule.metadata.api.model.MetadataType;
import org.mule.tooling.client.api.extension.model.DisplayModel;
import org.mule.tooling.client.api.extension.model.ExpressionSupport;
import org.mule.tooling.client.api.extension.model.LayoutModel;
import org.mule.tooling.client.api.extension.model.ParameterDslConfiguration;
import org.mule.tooling.client.api.extension.model.StereotypeModel;
import org.mule.tooling.client.api.extension.model.metadata.MetadataKeyPartModel;
import org.mule.tooling.client.api.extension.model.value.ValueProviderModel;

import java.util.List;
import java.util.Optional;

import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

/**
 * A parameter of a ComponentModel or Configuration
 * <p>
 * A parameter provides a name, a type and a default value.
 * </p>
 * It can apply either to a ConfigurationModel or a ComponentModel.
 *
 * @since 1.0
 */
public final class ParameterModel {

  private String name;
  private String description;
  private DisplayModel displayModel;
  private MetadataType type;
  private boolean hasDynamicType;
  private boolean required;
  private boolean isConfigOverride;
  private ExpressionSupport expressionSupport;
  private Object defaultValue;
  private ParameterRole role;
  private ParameterDslConfiguration dslConfiguration;
  private LayoutModel layoutModel;
  private MetadataKeyPartModel metadataKeyPartModel;
  private ValueProviderModel valueProviderModel;
  private List<StereotypeModel> allowedStereotypes;

  // Just needed in order to serialize this object
  private ParameterModel() {}

  public ParameterModel(String name,
                        String description,
                        DisplayModel displayModel,
                        MetadataType type,
                        boolean hasDynamicType,
                        boolean required,
                        boolean isConfigOverride,
                        ExpressionSupport expressionSupport,
                        Object defaultValue,
                        ParameterRole role,
                        ParameterDslConfiguration dslConfiguration,
                        LayoutModel layoutModel,
                        MetadataKeyPartModel metadataKeyPartModel,
                        ValueProviderModel valueProviderModel,
                        List<StereotypeModel> allowedStereotypes) {
    this.name = name;
    this.description = description;
    this.displayModel = displayModel;
    this.type = type;
    this.hasDynamicType = hasDynamicType;
    this.required = required;
    this.isConfigOverride = isConfigOverride;
    this.expressionSupport = expressionSupport;
    this.defaultValue = defaultValue;
    this.role = role;
    this.dslConfiguration = dslConfiguration;
    this.layoutModel = layoutModel;
    this.metadataKeyPartModel = metadataKeyPartModel;
    this.valueProviderModel = valueProviderModel;
    this.allowedStereotypes = allowedStereotypes;
  }

  public String getName() {
    return name;
  }

  public String getDescription() {
    return description;
  }

  public Optional<DisplayModel> getDisplayModel() {
    return ofNullable(displayModel);
  }

  public MetadataType getType() {
    return type;
  }

  public boolean hasDynamicType() {
    return hasDynamicType;
  }

  public boolean isRequired() {
    return required;
  }

  public boolean isOverrideFromConfig() {
    return isConfigOverride;
  }

  public ExpressionSupport getExpressionSupport() {
    return expressionSupport;
  }

  public Object getDefaultValue() {
    return defaultValue;
  }

  public ParameterDslConfiguration getDslConfiguration() {
    return dslConfiguration;
  }

  public ParameterRole getRole() {
    return role;
  }

  public Optional<LayoutModel> getLayoutModel() {
    return ofNullable(layoutModel);
  }

  public Optional<MetadataKeyPartModel> getMetadataKeyPartModel() {
    return ofNullable(metadataKeyPartModel);
  }

  public Optional<ValueProviderModel> getValueProviderModel() {
    return ofNullable(valueProviderModel);
  }

  public List<StereotypeModel> getAllowedStereotypes() {
    return allowedStereotypes;
  }

  @Override
  public int hashCode() {
    return this.name.hashCode();
  }

  @Override
  public boolean equals(Object obj) {
    return this.getClass().isInstance(obj) && this.name.equals(((ParameterModel) obj).getName());
  }

  @Override
  public String toString() {
    return ReflectionToStringBuilder.toString(this);
  }
}
