/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.tryit;

import org.mule.tooling.client.api.exception.NoSuchApplicationException;

import java.util.List;

/**
 * Service to do try it functionality over an application. The service will allow to enable try it feature over an application,
 * retrieve the collected components notifications and disable the feature.
 * <p/>
 * Applications should be deployed in order to be enabled for try it.
 * <p/>
 * It is up to the clients of the service to disable the feature, if not the try it job on Mule Agent will dispose the resources
 * and stop the feature on Mule Agent side.
 * <p/>
 * The service works as a client for the Mule Agent that buffers the components notifications and each invocation to
 * {@link #consume(String, int)} will update a heartbeat in Mule Agent.
 *
 * @since 1.0
 */
public interface TryItService {

  /**
   * Enables the try it functionality for the given application. Once enable it, the {@link #consume(String, int)} method would
   * retrieve the notifications collected
   *
   * @param applicationName a name to be used when deploying the application for collecting notifications
   * @throws NoSuchApplicationException if the application to be enable for try it is not already deployed.
   */
  void enable(String applicationName) throws NoSuchApplicationException;

  /**
   * Consumes the components notifications that were buffered by the Mule Agent until this moment for the given application.
   *
   * @param applicationName the application name to get its notifications
   * @param chunkSize the maximum amount of notifications to be returned
   * @return {@link List} of {@link Notification notifications}
   */
  MessageHistory consume(String applicationName, int chunkSize);

  /**
   * Disables the try it functionality for the given application.
   *
   * @param applicationName the application name to be disabled
   */
  void disable(String applicationName);

}
