/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.extension.model.operation;

import static java.util.Optional.ofNullable;
import org.mule.tooling.client.api.extension.model.Stereotype;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;

import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

/**
 * A Route is a chain of components which don't exist directly on a Mule flow,
 * but are contained inside another owning component.
 * <p>
 * For example:
 * <p>
 * <ul>
 * <li>The processors inside the {@code &lt;foreach&gt;} component are part of a route</li>
 * <li>The processors inside the {@code &lt;when&gt;} block inside a choice router.</li>
 * </ul>
 * <p>
 * Each of the above constitute a {@code Route}. Several instances of those routes can exists
 * inside the same component instance. For example, the {@code when} route can be used many times
 * inside a choice router. On the other hand, the {@code otherwise} route inside the same router can
 * only exist up to one time, while it is also allowed to not exist at all. This {@link #getMinOccurs()}
 * and {@link #getMaxOccurs()} methods model that.
 * <p>
 * Finally, some scope have restrictions regarding which components can be used with them. For example,
 * {@code validation:all} only allows other validators to be placed inside. That's what the {@link #getAllowedStereotypes()}
 * method represents.
 *
 * @since 1.0
 */
public class RouteModel {

  private String name;
  private String description;
  private int minOccurs;
  private Integer maxOccurs;
  private Set<Stereotype> allowedStereotypes;
  private List<ParameterGroupModel> parameterGroupModels;

  // Just needed in order to serialize this object
  private RouteModel() {}

  public RouteModel(String name, String description, int minOccurs, Integer maxOccurs,
                    Set<Stereotype> allowedStereotypes,
                    List<ParameterGroupModel> parameterGroupModels) {
    this.name = name;
    this.description = description;
    this.minOccurs = minOccurs;
    this.maxOccurs = maxOccurs;
    this.allowedStereotypes = allowedStereotypes;
    this.parameterGroupModels = parameterGroupModels;
  }

  public String getName() {
    return name;
  }

  public String getDescription() {
    return this.description;
  }

  public void setDescription(String description) {
    this.description = description;
  }

  public int getMinOccurs() {
    return minOccurs;
  }

  public Optional<Integer> getMaxOccurs() {
    return ofNullable(maxOccurs);
  }

  public List<ParameterGroupModel> getParameterGroupModels() {
    return parameterGroupModels;
  }

  public Optional<Set<Stereotype>> getAllowedStereotypes() {
    return ofNullable(allowedStereotypes);
  }

  @Override
  public int hashCode() {
    return this.name.hashCode();
  }

  @Override
  public boolean equals(Object obj) {
    return this.getClass().isInstance(obj) && this.name.equals(((RouteModel) obj).getName());
  }

  @Override
  public String toString() {
    return ReflectionToStringBuilder.toString(this);
  }

}
