/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.configuration.agent;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;

import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;

import javax.net.ssl.SSLContext;
import java.net.URL;
import java.util.Optional;
import java.util.function.Supplier;

/**
 * Immutable implementation for {@link AgentConfiguration}.
 *
 * @since 1.0
 */
public class ImmutableAgentConfiguration implements AgentConfiguration {

  private Supplier<URL> toolingApiURLSupplier;
  private int defaultConnectionTimeout;
  private int defaultReadTimeout;
  private Optional<SSLContext> sslContext;

  public ImmutableAgentConfiguration(Supplier<URL> toolingApiURLSupplier, int defaultConnectionTimeout, int defaultReadTimeout,
                                     SSLContext sslContext) {
    requireNonNull(toolingApiURLSupplier, "toolingApiURLSupplier cannot be null");

    this.toolingApiURLSupplier = toolingApiURLSupplier;
    this.defaultConnectionTimeout = defaultConnectionTimeout;
    this.defaultReadTimeout = defaultReadTimeout;
    this.sslContext = ofNullable(sslContext);
  }

  @Override
  public Supplier<URL> getToolingApiURLSupplier() {
    return toolingApiURLSupplier;
  }

  @Override
  public int getDefaultConnectTimeout() {
    return defaultConnectionTimeout;
  }

  @Override
  public int getDefaultReadTimeout() {
    return defaultReadTimeout;
  }

  @Override
  public Optional<SSLContext> getSSLContext() {
    return this.sslContext;
  }

  @Override
  public String toString() {
    return format("%s{defaultConnectionTimeout=%sms, defaultReadTimeout=%sms}",
                  this.getClass().getName(), defaultConnectionTimeout, defaultReadTimeout);
  }

}
