/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.runtime.process.controller;

import static java.io.File.separator;
import static java.util.Objects.requireNonNull;

import java.io.File;

/**
 * Service wrapper for Mule Runtime Standalone EE.
 * <p>
 * It knows distribution's folders layout and how allows to run Tanuki service wrapper commands to operate the
 * Mule Runtime.
 * See {@link org.mule.tooling.runtime.process.controller.command.MuleCommand} for more information about how to
 * operate a Mule Runtime.
 *
 * @since 4.0
 */
public class MuleServiceWrapper {

  public static final String MULE_HOME = "MULE_HOME";

  private static final String BIN_FOLDER = "bin";
  private final File muleHome;
  private final Long timeout;
  private String muleBinary;

  /**
   * Creates an instance of the service wrapper.
   *
   * @param muleHome          {@link File} location of the local Mule Runtime {@code MULE_HOME}
   * @param timeout           milliseconds to wait for Mule Runtime operations to be completed.
   * @param muleBinaryCommand {@link String} defining the command.
   */
  MuleServiceWrapper(File muleHome, long timeout, String muleBinaryCommand) {
    requireNonNull(muleHome, "muleHome cannot be null");

    this.muleHome = muleHome;
    this.muleBinary = muleHome + separator + BIN_FOLDER + separator + muleBinaryCommand;
    this.timeout = timeout;

    if (!(muleHome.exists() && muleHome.isDirectory())) {
      throw new IllegalStateException("muleHome directory is not referencing to a valid Mule Runtime directory structure");
    }
  }

  public String getMuleBinaryCommand() {
    return this.muleBinary;
  }

  public Long getProcessInvocationTimeout() {
    return this.timeout;
  }

  public File getMuleHome() {
    return this.muleHome;
  }

}
