/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.event.model;

import static java.util.Objects.requireNonNull;

import java.util.HashMap;
import java.util.Map;

public class EventModel {

  private MessageModel message;
  private Map<String, TypedValueModel> variables = new HashMap<>();
  private ErrorModel error;

  // Used by json serializer/deserialzer
  public EventModel() {}

  private EventModel(MessageModel message, Map<String, TypedValueModel> variables, ErrorModel error) {
    requireNonNull(message, "message cannot be null");

    this.message = message;
    this.variables = variables;
    this.error = error;
  }

  public MessageModel getMessage() {
    return message;
  }

  public Map<String, TypedValueModel> getVariables() {
    return variables;
  }

  public ErrorModel getError() {
    return error;
  }

  public static Builder builder() {
    return new Builder();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    EventModel that = (EventModel) o;

    if (!message.equals(that.message)) {
      return false;
    }
    if (variables != null ? !variables.equals(that.variables) : that.variables != null) {
      return false;
    }
    return error != null ? error.equals(that.error) : that.error == null;
  }

  @Override
  public int hashCode() {
    int result = message.hashCode();
    result = 31 * result + (variables != null ? variables.hashCode() : 0);
    result = 31 * result + (error != null ? error.hashCode() : 0);
    return result;
  }

  public static class Builder {

    private MessageModel message;
    private Map<String, TypedValueModel> variables;
    private ErrorModel error;

    private Builder() {}

    public Builder withMessage(MessageModel message) {
      this.message = message;
      return this;
    }

    public Builder withVariables(Map<String, TypedValueModel> variables) {
      this.variables = variables;
      return this;
    }

    public Builder withError(ErrorModel error) {
      this.error = error;
      return this;
    }

    public EventModel build() {
      return new EventModel(this.message, this.variables, this.error);
    }
  }


}
