/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.agent.rest.client;

import static java.util.Collections.singletonMap;
import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.assertThat;
import static org.mule.tck.junit4.matcher.metadata.MetadataKeyResultSuccessMatcher.isSuccess;

import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataDescriptor;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.tck.junit4.rule.DynamicPort;

import java.io.File;
import java.net.URISyntaxException;
import java.util.Map;

import com.icegreen.greenmail.util.GreenMail;
import com.icegreen.greenmail.util.ServerSetup;
import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

@Feature("Remote Runtime Tooling Service")
@Story("Metadata")
public class MetadataTestCase extends AbstractRestAgentToolingServiceTestCase {

  private static final String MAIL_SERVER_PORT_ARG = "emailServerPort";
  private static final String EMAIL_FLOW = "email:flow";
  private static final String EMAIL_CONFIG = "emailConfig";

  @Rule
  public DynamicPort emailPort = new DynamicPort("emailPort");

  private GreenMail greenMail;

  protected Map<String, String> getDeploymentProperties() {
    return singletonMap(MAIL_SERVER_PORT_ARG, emailPort.getValue());
  }

  @Before
  public void setUpEmailServer() {
    ServerSetup serverSetup = new ServerSetup(emailPort.getNumber(), null, "imap");
    greenMail = new GreenMail(serverSetup);
    greenMail.setUser("foo", "pwd");
    greenMail.start();
  }

  @After
  public void stopEmailServer() {
    if (greenMail != null) {
      greenMail.stop();
    }
  }

  @Test
  @Description("Checks the Metadata resolution from Mule Rutime Agent using the RuntimeToolingService")
  public void getMetadataTest() throws URISyntaxException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailAppLocation = new File(targetTestClassesFolder, "applications/email");

    String applicationId = restAgentToolingService.deployApplication(null, emailAppLocation, getDeploymentProperties());

    MetadataResult<ComponentMetadataDescriptor<OperationModel>> result =
        restAgentToolingService.getOperationMetadata(applicationId, Location.builder().globalName(EMAIL_FLOW).addProcessorsPart()
            .addIndexPart(0).build().toString(), 10000);

    assertThat(result, isSuccess());
    assertThat(result.get().getModel().getName(), equalTo("listImap"));

    result = restAgentToolingService
        .getOperationMetadata(applicationId,
                              Location.builder().globalName(EMAIL_FLOW).addProcessorsPart().addIndexPart(3).build().toString(),
                              10000);

    assertThat(result, isSuccess());
    assertThat(result.get().getModel().getName(), equalTo("markAsDeleted"));
  }

  @Test
  @Description("Checks the Metadata resolution for keys on domain from Mule Runtime Agent using the RuntimeToolingService")
  public void getMetadataKeysFromDomain() throws URISyntaxException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailDomainLocation = new File(targetTestClassesFolder, "domains/email-domain-exploded");

    String domainId = restAgentToolingService.deployDomain(null, emailDomainLocation, getDeploymentProperties());

    MetadataResult<MetadataKeysContainer> result =
        restAgentToolingService.getMetadataKeysDomain(domainId, Location.builder().globalName(EMAIL_CONFIG).build().toString(),
                                                      10000);

    assertThat(result, isSuccess());
  }

}
