/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.agent.rest.client;

import static java.util.Collections.emptyMap;
import static java.util.UUID.randomUUID;
import static java.util.concurrent.TimeUnit.SECONDS;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.fail;
import static org.junit.rules.ExpectedException.none;

import org.mule.tooling.client.api.exception.TimeoutException;
import org.mule.tooling.client.test.utils.probe.JUnitLambdaProbe;
import org.mule.tooling.client.test.utils.probe.PollingProber;

import java.io.File;
import java.util.List;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

public class DeployApplicationTimeOutTestCase extends AbstractDeploymentTestCase {

  public static final String MULE_TOOLING_APPS = "/.mule/tooling/apps";
  @Rule
  public ExpectedException expectedException = none();
  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Before
  public void beforeTest() {
    setUpEmailServer();
  }

  @After
  public void afterTest() {
    stopEmailServer();
  }

  @Test
  public void doDeployApplicationTimeout() throws Exception {
    RestAgentToolingService timeOutRestAgentToolingService =
        createRestAgentToolingService(SECONDS.toMillis(AGENT_CONNECT_TIMEOUT_SECS), 100);

    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailAppLocation = new File(targetTestClassesFolder, "applications/email");

    String uuid = randomUUID().toString();
    try {
      timeOutRestAgentToolingService.deployApplication(uuid, emailAppLocation, emptyMap());
      fail("A timeoutException should be thrown");
    } catch (TimeoutException e) {
      // Ok
    }

    new PollingProber(SECONDS.toMillis(AGENT_READ_TIMEOUT_SECS), 200)
        .check(new JUnitLambdaProbe(() -> {
          List<String> listApplicationsId = restAgentToolingService.listApplicationsId();
          return listApplicationsId.size() == 1;
        }));
    assertToolingApplicationDeployed(uuid);

    // Try to deploy again the same application with same UUID, should return the previous deployed application
    String deployedId = restAgentToolingService.deployApplication(uuid, emailAppLocation, emptyMap());
    assertThat(deployedId, equalTo(uuid));

    assertToolingApplicationDeployed(deployedId);
  }

  private void assertToolingApplicationDeployed(String uuid) {
    List<String> listApplicationsId = restAgentToolingService.listApplicationsId();
    assertThat(listApplicationsId, hasSize(1));
    assertThat(listApplicationsId, contains(equalTo(uuid)));
    File toolingApplications = new File(muleHome + MULE_TOOLING_APPS);
    assertThat(toolingApplications.exists(), is(true));
    assertThat(toolingApplications.list().length, is(1));
  }

}
