/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tck;

import static org.mule.maven.pom.parser.api.MavenPomParserProvider.discoverProvider;
import static org.mule.tck.ZipUtils.compress;

import static java.nio.file.Files.copy;
import static java.nio.file.Files.createDirectories;
import static java.nio.file.StandardCopyOption.COPY_ATTRIBUTES;
import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;
import static java.util.Arrays.asList;

import static org.apache.commons.io.FilenameUtils.getExtension;

import org.mule.maven.pom.parser.api.MavenPomParser;
import org.mule.maven.pom.parser.api.model.MavenPomModel;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

public class MavenTestUtils {

  private static final String POM = "pom";

  public static Collection<Path> installArtifact(Path artifactFile, Path repositoryLocation) throws IOException {
    String artifactExtension = getExtension(artifactFile.getFileName().toString());
    MavenPomParser parser = discoverProvider().createMavenPomParserClient(artifactFile);
    if (POM.equals(artifactExtension)) {
      return installArtifact(artifactFile, repositoryLocation, parser);
    } else {
      Path packagedArtifact = packageArtifact(artifactFile, parser.getModel());
      return installArtifact(packagedArtifact, repositoryLocation, parser);
    }
  }

  private static Path packageArtifact(Path explodedArtifactFile, MavenPomModel pomModel) throws IOException {
    String fileNameInRepo = pomModel.getArtifactId()
        + "-" + pomModel.getVersion()
        + (pomModel.getPackaging() != null && !pomModel.getPackaging().equalsIgnoreCase("jar") ? "-" + pomModel.getPackaging()
            : "")
        + ".jar";
    Path compressedFile = explodedArtifactFile.resolve(fileNameInRepo);

    try (var recursiveResources = Files.walk(explodedArtifactFile)) {
      compress(compressedFile, recursiveResources
          .filter(Files::isRegularFile)
          .map(f -> new ZipUtils.ZipResource(f.toAbsolutePath().toString(),
                                             getZipEntryName(explodedArtifactFile, f)))
          .toArray(ZipUtils.ZipResource[]::new));
    }

    return compressedFile;
  }

  private static String getZipEntryName(Path baseDir, Path entryFile) {
    // https://pkware.cachefly.net/webdocs/casestudies/APPNOTE.TXT - 4.4.17.1
    // The path stored MUST NOT contain a drive or device letter, or a leading slash. All slashes MUST be forward slashes '/' as
    // opposed to backwards slashes '\'
    return baseDir.toUri().relativize(entryFile.toUri()).getPath();
  }

  private static Collection<Path> installArtifact(Path artifactFile, Path repositoryLocation, MavenPomParser parser)
      throws IOException {
    MavenPomModel pomModel = parser.getModel();
    List<String> artifactLocationInRepo = new ArrayList<>(asList(pomModel.getGroupId().split("\\.")));
    artifactLocationInRepo.add(pomModel.getArtifactId());
    artifactLocationInRepo.add(pomModel.getVersion());

    Path pathToArtifactLocationInRepo = repositoryLocation;
    for (String artifactLocationInRepoPart : artifactLocationInRepo) {
      pathToArtifactLocationInRepo = pathToArtifactLocationInRepo.resolve(artifactLocationInRepoPart);
    }

    createDirectories(pathToArtifactLocationInRepo);

    Path repoArtifactFile = pathToArtifactLocationInRepo.resolve(artifactFile.getFileName());
    copy(artifactFile, repoArtifactFile, COPY_ATTRIBUTES, REPLACE_EXISTING);

    // Copy the pom without the classifier.
    String pomFileName = artifactFile.getFileName().toString().replaceFirst("(.*\\.[0-9]*\\.[0-9]*\\.?[0-9]?).*", "$1") + ".pom";
    Path repoPomFile = pathToArtifactLocationInRepo.resolve(pomFileName);
    copy(parser.getModel().getPomFile().get().toPath(), repoPomFile, COPY_ATTRIBUTES, REPLACE_EXISTING);

    return asList(repoArtifactFile, repoPomFile);
  }

}
