/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.artifact.builder;

import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.tck.ZipUtils.compress;

import static java.io.File.separator;
import static java.nio.file.Files.newOutputStream;
import static java.nio.file.Files.writeString;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;

import static com.vdurmont.semver4j.Semver.SemverType.LOOSE;
import static org.apache.commons.io.FilenameUtils.getBaseName;
import static org.apache.commons.io.FilenameUtils.getName;
import static org.apache.commons.lang3.StringUtils.isEmpty;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.core.Is.is;

import org.mule.runtime.api.deployment.meta.MuleArtifactLoaderDescriptor;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.tck.ZipUtils.ZipResource;

import java.io.IOException;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.Properties;

import com.vdurmont.semver4j.Semver;

/**
 * Defines a builder to create files for mule artifacts.
 * <p/>
 * Instances can be configured using the methods that follow the builder pattern until the artifact file is accessed. After that
 * point, builder methods will fail to update the builder state.
 *
 * @param <T> class of the implementation builder
 */
public abstract class AbstractArtifactFileBuilder<T extends AbstractArtifactFileBuilder<T>>
    extends AbstractDependencyFileBuilder<T>
    implements TestArtifactDescriptor {

  protected static final String CLASSLOADER_MODEL_JSON_DESCRIPTOR = "classloader-model.json";
  protected static final String CLASSLOADER_MODEL_JSON_DESCRIPTOR_LOCATION =
      Paths.get("META-INF", "mule-artifact", CLASSLOADER_MODEL_JSON_DESCRIPTOR).toString();

  protected static final Semver CLASS_LOADER_MODEL_VERSION_110 = new Semver("1.1.0", LOOSE);
  protected static final String CLASS_LOADER_MODEL_VERSION_120 = "1.2.0";

  private final boolean upperCaseInExtension;
  protected Path artifactFile;
  protected List<ZipResource> resources = new LinkedList<>();
  protected boolean corrupted;

  /**
   * Creates a new builder
   *
   * @param artifactId           artifact identifier. Non empty.
   * @param upperCaseInExtension whether the extension is in uppercase
   */
  public AbstractArtifactFileBuilder(String artifactId, boolean upperCaseInExtension) {
    super(artifactId);
    this.upperCaseInExtension = upperCaseInExtension;
    checkArgument(!isEmpty(artifactId), "ID cannot be empty");
  }

  /**
   * Creates a new builder
   *
   * @param artifactId artifact identifier. Non empty.
   */
  public AbstractArtifactFileBuilder(String artifactId) {
    this(artifactId, false);
  }

  /**
   * Template method to redefine the file extension
   *
   * @return the file extension of the file name for the artifact.
   */
  protected String getFileExtension() {
    return ".jar";
  }

  /**
   * Creates a new builder from another instance.
   *
   * @param source instance used as template to build the new one. Non null.
   */
  public AbstractArtifactFileBuilder(T source) {
    this(source.getArtifactId(), source);
  }

  /**
   * Create a new builder from another instance and different ID.
   *
   * @param id     artifact identifier. Non empty.
   * @param source instance used as template to build the new one. Non null.
   */
  public AbstractArtifactFileBuilder(String id, T source) {
    this(id);
    this.resources.addAll(source.resources);
    this.corrupted = source.corrupted;
  }

  /**
   * Adds a resource file to the artifact folder.
   *
   * @param resourceFile class file from a external file or test resource. Non empty.
   * @param targetFile   name to use on the added resource. Non empty.
   * @return the same builder instance
   */
  public T usingResource(String resourceFile, String targetFile) {
    checkImmutable();
    checkArgument(!isEmpty(resourceFile), "Resource file cannot be empty");
    resources.add(new ZipResource(resourceFile, targetFile));

    return getThis();
  }

  /**
   * Adds a jar file to the artifact lib folder.
   *
   * @param jarFile jar file from a external file or test resource.
   * @return the same builder instance
   */
  public T usingLibrary(String jarFile) {
    checkImmutable();
    checkArgument(!isEmpty(jarFile), "Jar file cannot be empty");
    resources.add(new ZipResource(jarFile, "lib/" + getName(jarFile)));

    return getThis();
  }

  /**
   * Adds a jar file to the artifact lib folder.
   *
   * @param jarFile jar file from a external file or test resource.
   * @return the same builder instance
   */
  public T usingLibraries(Path... jarFiles) {
    checkImmutable();

    for (Path jarFile : jarFiles) {
      checkArgument(!isEmpty(jarFile.toAbsolutePath().toString()), "Jar file cannot be empty");
      resources.add(new ZipResource(jarFile.toAbsolutePath().toString(), "lib/" + getName(jarFile.toAbsolutePath().toString())));

    }

    return getThis();
  }

  /**
   * Adds a class file to the artifact classes folder.
   *
   * @param classFile class file to include. Non null.
   * @param alias     path where the file must be added inside the app file
   * @return the same builder instance
   */
  public T containingClass(Path classFile, String alias) {
    checkImmutable();
    requireNonNull(classFile, "Class file cannot be null");
    resources.add(new ZipResource(classFile.toAbsolutePath().toString(), alias));

    return getThis();
  }

  /**
   * Adds a resource file to the plugin root folder.
   *
   * @param resourceFile resource file from a external file or test resource.
   * @return the same builder instance
   */
  public T containingResource(String resourceFile, String alias) {
    checkImmutable();
    checkArgument(!isEmpty(resourceFile), "Resource file cannot be empty");
    resources.add(new ZipResource(resourceFile, alias));

    return getThis();
  }

  /**
   * Indicates that the generated artifact file must be a corrupted ZIP.
   *
   * @return the same builder instance
   */
  public T corrupted() {
    checkImmutable();
    this.corrupted = true;

    return getThis();
  }

  @Override
  public String getId() {
    return getBaseName(getArtifactFileName());
  }

  @Override
  public String getZipPath() {
    if (artifactFile == null) {
      throw new IllegalStateException("Must generate the artifact file before invoking this method");
    }
    return separator + artifactFile.getFileName().toString();
  }

  @Override
  public String getDeployedPath() {
    if (artifactFile == null) {
      throw new IllegalStateException("Must generate the artifact file before invoking this method");
    }
    if (corrupted) {
      return artifactFile.getFileName().toString();
    } else {
      return getBaseName(artifactFile.getFileName().toString());
    }
  }

  @Override
  public Path getArtifactFile() throws IOException {
    if (artifactFile == null) {
      try {
        String fileName = getArtifactFileName();
        final Path tempFile = getTempFolder().resolve(fileName);
        tempFile.toFile().deleteOnExit();

        if (corrupted) {
          buildBrokenJarFile(tempFile);
        } else {
          final List<ZipResource> zipResources = new LinkedList<>(resources);
          zipResources.add(new ZipResource(getArtifactPomFile().toAbsolutePath().toString(),
                                           getArtifactFileBundledPomPartialUrl()));
          zipResources.add(new ZipResource(getArtifactPomPropertiesFile().toAbsolutePath().toString(),
                                           getArtifactFileBundledPomPropertiesPartialUrl()));
          zipResources.addAll(getCustomResources());
          compress(tempFile, zipResources.toArray(new ZipResource[0]));
        }

        artifactFile = tempFile.toAbsolutePath();
      } catch (IOException e) {
        throw new MuleRuntimeException(e);
      }
    }

    return artifactFile;
  }

  private String getArtifactFileName() {
    String fileName = getArtifactId();
    String artifactNameSeparator = "-";
    if (getVersion() != null) {
      fileName = fileName + artifactNameSeparator + getVersion();
    }

    if (getClassifier() != null) {
      fileName = fileName + artifactNameSeparator + getClassifier();
    }

    fileName = fileName + ((upperCaseInExtension) ? getFileExtension().toUpperCase() : getFileExtension().toLowerCase());
    return fileName;
  }

  protected final void checkImmutable() {
    assertThat("Cannot change attributes once the artifact file was built", artifactFile, is(nullValue()));
  }

  protected ZipResource createPropertiesFile(Properties props, String propertiesFileName, String zipAlias) throws IOException {
    ZipResource result = null;

    if (!props.isEmpty()) {
      final Path applicationPropertiesFile = getTempFolder().resolve(propertiesFileName);
      applicationPropertiesFile.toFile().deleteOnExit();
      createPropertiesFile(applicationPropertiesFile, props);

      result = new ZipResource(applicationPropertiesFile.toAbsolutePath().toString(), zipAlias);
    }

    return result;
  }

  protected void createPropertiesFile(Path file, Properties props) {
    try (OutputStream out = newOutputStream(file)) {
      props.store(out, "Generated application properties");
    } catch (Exception e) {
      throw new IllegalStateException("Cannot create properties", e);
    }
  }

  private void buildBrokenJarFile(Path tempFile) throws UncheckedIOException {
    try {
      writeString(tempFile, "This content represents invalid compressed data");
    } catch (IOException e) {
      throw new UncheckedIOException(e);
    }
  }

  /**
   * @return a collection with custom {@link ZipResource}s to add to the artifact file.
   */
  protected abstract List<ZipResource> getCustomResources() throws IOException;

  /**
   * @return the descriptor loader for the artifact.
   */
  protected Optional<MuleArtifactLoaderDescriptor> getBundleDescriptorLoader() {
    return empty();
  }

  @Override
  public String getGroupId() {
    return (String) getBundleDescriptorLoader().map(descriptorLoader -> descriptorLoader.getAttributes().get("groupId"))
        .orElse(super.getGroupId());
  }

  @Override
  public String getArtifactId() {
    return (String) getBundleDescriptorLoader().map(descriptorLoader -> descriptorLoader.getAttributes().get("artifactId"))
        .orElse(super.getArtifactId());
  }

  @Override
  public String getClassifier() {
    return (String) getBundleDescriptorLoader().map(descriptorLoader -> descriptorLoader.getAttributes().get("classifier"))
        .orElse(super.getClassifier());
  }

  @Override
  public String getType() {
    return (String) getBundleDescriptorLoader().map(descriptorLoader -> descriptorLoader.getAttributes().get("type"))
        .orElse(super.getType());
  }

  @Override
  public String getVersion() {
    return (String) getBundleDescriptorLoader().map(descriptorLoader -> descriptorLoader.getAttributes().get("version"))
        .orElse(super.getVersion());
  }
}
