/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tck.junit5;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Annotation to mark fields that should have classloader cleanup applied.
 *
 * <p>
 * When applied to any {@link java.io.Closeable} ClassLoader field, the {@link ClassLoaderCleanupExtension} will automatically:
 * <ol>
 * <li>Store the original context classloader before test execution</li>
 * <li>After test execution: close the annotated classloader and restore the original context classloader</li>
 * </ol>
 *
 * <p>
 * Supports both instance fields (cleaned after each test method) and static fields (cleaned after all test methods).
 *
 * <p>
 * Example usage:
 *
 * <pre>
 *
 * {
 *   &#64;code
 *   &#64;ExtendWith(ClassLoaderCleanupExtension.class)
 *   public class MyTestCase {
 *
 *     &#64;ClassLoaderCleanup
 *     private URLClassLoader customClassLoader;
 *
 *     &#64;ClassLoaderCleanup
 *     private static MyCloseableClassLoader staticClassLoader;
 *
 *     &#64;BeforeEach
 *     public void setup() {
 *       customClassLoader = new URLClassLoader(new URL[0], Thread.currentThread().getContextClassLoader());
 *     }
 *     // Tests will automatically have the custom classloader set as context classloader
 *     // and cleanup will be handled automatically
 *   }
 * }
 * </pre>
 */
@Target(ElementType.FIELD)
@Retention(RetentionPolicy.RUNTIME)
public @interface ClassLoaderCleanup {
}
