/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tck.junit4.rule;

import static org.junit.Assume.assumeThat;

import org.hamcrest.Matcher;
import org.junit.rules.ExternalResource;

/**
 * Checks that the JVM version meets certain criteria. If not, the test case is ignored.
 * <p>
 * Example usage:
 *
 * <pre>
 *
 * // Assume JVM version is at least 11
 * &#64;Rule
 * public JvmVersion jvmVersion = JvmVersion.atLeast(11);
 *
 * // Assume JVM version is exactly 17
 * &#64;Rule
 * public JvmVersion jvmVersion = JvmVersion.is(17);
 *
 * // Assume JVM version is less than 21
 * &#64;Rule
 * public JvmVersion jvmVersion = JvmVersion.lessThan(21);
 *
 * // Custom matcher
 * &#64;Rule
 * public JvmVersion jvmVersion = new JvmVersion(anyOf(equalTo(11), equalTo(17)));
 * </pre>
 *
 * @since 4.11
 */
public class JvmVersion extends ExternalResource {

  private final Matcher<Integer> versionMatcher;
  private final String description;

  /**
   * Constructs a JUnit Rule to assume the JVM version matches the provided matcher.
   *
   * @param versionMatcher a Hamcrest matcher to evaluate the JVM version against.
   */
  public JvmVersion(Matcher<Integer> versionMatcher) {
    this(versionMatcher, null);
  }

  /**
   * Constructs a JUnit Rule to assume the JVM version matches the provided matcher.
   *
   * @param versionMatcher a Hamcrest matcher to evaluate the JVM version against.
   * @param description    custom description for the assumption message.
   */
  public JvmVersion(Matcher<Integer> versionMatcher, String description) {
    this.versionMatcher = versionMatcher;
    this.description = description;
  }

  @Override
  protected void before() throws Throwable {
    super.before();

    int currentVersion = getCurrentJvmVersion();
    String message = description != null ? description
        : "JVM version " + currentVersion + " does not meet the required criteria. Ignoring test.";

    assumeThat(message, currentVersion, versionMatcher);
  }

  /**
   * Creates a rule that assumes the JVM version is at least the specified version.
   *
   * @param minVersion the minimum required JVM version (inclusive)
   * @return a new JvmVersion rule
   */
  public static JvmVersion atLeast(int minVersion) {
    return new JvmVersion(
                          org.hamcrest.Matchers.greaterThanOrEqualTo(minVersion),
                          "JVM version must be at least " + minVersion + ". Ignoring test.");
  }

  /**
   * Creates a rule that assumes the JVM version is exactly the specified version.
   *
   * @param version the exact required JVM version
   * @return a new JvmVersion rule
   */
  public static JvmVersion is(int version) {
    return new JvmVersion(
                          org.hamcrest.Matchers.equalTo(version),
                          "JVM version must be " + version + ". Ignoring test.");
  }

  /**
   * Creates a rule that assumes the JVM version is less than the specified version.
   *
   * @param maxVersion the maximum JVM version (exclusive)
   * @return a new JvmVersion rule
   */
  public static JvmVersion lessThan(int maxVersion) {
    return new JvmVersion(
                          org.hamcrest.Matchers.lessThan(maxVersion),
                          "JVM version must be less than " + maxVersion + ". Ignoring test.");
  }

  /**
   * Creates a rule that assumes the JVM version is at most the specified version.
   *
   * @param maxVersion the maximum required JVM version (inclusive)
   * @return a new JvmVersion rule
   */
  public static JvmVersion atMost(int maxVersion) {
    return new JvmVersion(
                          org.hamcrest.Matchers.lessThanOrEqualTo(maxVersion),
                          "JVM version must be at most " + maxVersion + ". Ignoring test.");
  }

  /**
   * Creates a rule that assumes the JVM version is within the specified range.
   *
   * @param minVersion the minimum required JVM version (inclusive)
   * @param maxVersion the maximum required JVM version (inclusive)
   * @return a new JvmVersion rule
   */
  public static JvmVersion between(int minVersion, int maxVersion) {
    return new JvmVersion(
                          org.hamcrest.Matchers.allOf(
                                                      org.hamcrest.Matchers.greaterThanOrEqualTo(minVersion),
                                                      org.hamcrest.Matchers.lessThanOrEqualTo(maxVersion)),
                          "JVM version must be between " + minVersion + " and " + maxVersion + ". Ignoring test.");
  }

  /**
   * Gets the current JVM major version.
   *
   * @return the major version of the JVM (e.g., 8, 11, 17, 21)
   */
  public static int getCurrentJvmVersion() {
    String version = System.getProperty("java.version");
    if (version.startsWith("1.")) {
      // Java 8 or earlier (e.g., "1.8.0_292")
      version = version.substring(2, 3);
    } else {
      // Java 9+ (e.g., "11.0.12", "17.0.1")
      int dotIndex = version.indexOf('.');
      if (dotIndex != -1) {
        version = version.substring(0, dotIndex);
      }
    }
    return Integer.parseInt(version);
  }
}

