/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tck.junit5;

import static java.lang.String.format;
import static java.lang.String.valueOf;
import static java.lang.System.clearProperty;
import static java.lang.System.getProperty;
import static java.lang.System.setProperty;

import org.mule.tck.junit4.rule.FreePortFinder;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Allows annotating fields of type {@link Integer} with the annotation {@link DynamicPort}. The annotated fields will have a port
 * that can be bound during the test, and a system property will be set with that value and the name provided in the property.
 */
public class DynamicPortExtension extends AbstractResourceExtension<DynamicPort, Integer> {

  public static final String MIN_PORT_SYSTEM_PROPERTY = "mule.test.minPort";
  public static final String MAX_PORT_SYSTEM_PROPERTY = "mule.test.maxPort";

  private static final int DEFAULT_MIN_PORT = 10000;
  private static final int DEFAULT_MAX_PORT = 40000;

  protected static FreePortFinder freePortFinder;

  static {
    int minPort = DEFAULT_MIN_PORT;
    int maxPort = DEFAULT_MAX_PORT;

    String propertyValue = System.getProperty(MIN_PORT_SYSTEM_PROPERTY);
    if (propertyValue != null) {
      minPort = Integer.parseInt(propertyValue);
    }

    propertyValue = System.getProperty(MAX_PORT_SYSTEM_PROPERTY);
    if (propertyValue != null) {
      maxPort = Integer.parseInt(propertyValue);
    }

    if (minPort > maxPort) {
      throw new IllegalArgumentException(format("Min port '%s' must be less than max port '%s'", minPort, maxPort));
    }

    freePortFinder = new FreePortFinder(minPort, maxPort);
  }

  private final Map<String, String> propertiesToRestore = new ConcurrentHashMap<>();

  public DynamicPortExtension() {
    super(DynamicPort.class);
  }

  @Override
  protected Integer createResource(DynamicPort annotation) {
    var systemProperty = annotation.systemProperty();
    String oldPropertyValue = getProperty(systemProperty);
    if (oldPropertyValue != null) {
      propertiesToRestore.put(systemProperty, oldPropertyValue);
    }
    Integer portNumber = freePortFinder.find();
    setProperty(systemProperty, valueOf(portNumber));
    return portNumber;
  }

  @Override
  protected void disposeResource(DynamicPort annotation, Integer portNumber) {
    var systemProperty = annotation.systemProperty();
    String oldValue = propertiesToRestore.remove(systemProperty);
    if (oldValue != null) {
      setProperty(systemProperty, oldValue);
    } else {
      clearProperty(systemProperty);
    }
    freePortFinder.releasePort(portNumber);
  }
}
