/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tck.junit5;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Repeatable;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import org.junit.jupiter.api.extension.ExtendWith;

/**
 * Annotation to set and manage system properties in JUnit 5 tests. The system property will be set before each test and restored
 * after each test.
 *
 * <p>
 * Example usage:
 * </p>
 *
 * <pre>
 *
 * {
 *   &#64;code
 *   &#64;SystemProperty(name = "my.property", value = "test-value")
 *   &#64;SystemProperty(name = "another.property", value = "another-value")
 *   class MyTest {
 *     // tests here
 *   }
 * }
 * </pre>
 */
@Target({ElementType.TYPE, ElementType.METHOD, ElementType.FIELD})
@Retention(RetentionPolicy.RUNTIME)
@Documented
@Repeatable(SystemProperties.class)
@ExtendWith(SystemPropertyExtension.class)
public @interface SystemProperty {

  /**
   * The name of the system property to set.
   */
  String name();

  /**
   * The value of the system property to set.
   */
  String value();
}
