/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.test.components.tracing;

import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.ALWAYS_ON_SAMPLER;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_EXPORTER_ENABLED;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_EXPORTER_TRACES_SAMPLER;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_TRACING_LEVEL_CONFIGURATION_FILE_NAME;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_TRACING_LEVEL_CONFIGURATION_FILE_PATH;
import static org.mule.runtime.tracing.level.impl.config.FileTracingLevelConfiguration.CONFIGURATION_FILE_NAME;

import static java.lang.Boolean.TRUE;
import static java.lang.String.format;

import org.mule.functional.junit4.MuleArtifactFunctionalTestCase;
import org.mule.tck.junit4.rule.SystemProperty;

import java.io.File;
import java.net.URL;

import org.junit.Rule;

public abstract class OpenTelemetryTracingTestCase extends MuleArtifactFunctionalTestCase {

  @Rule
  public SystemProperty enableTracing = new SystemProperty(MULE_OPEN_TELEMETRY_EXPORTER_ENABLED, TRUE.toString());
  @Rule
  public SystemProperty doNotSample = new SystemProperty(MULE_OPEN_TELEMETRY_EXPORTER_TRACES_SAMPLER, ALWAYS_ON_SAMPLER);
  @Rule
  public SystemProperty tracingLevelConfigurationFileNameProperty;
  @Rule
  public SystemProperty tracingLevelConfigurationFilePathProperty;

  /**
   * Configures tracing level for the test by setting the appropriate SystemProperty fields. Call this method from your test
   * constructor to initialize tracing configuration using filesystem-level configuration.
   *
   * @param configurationFolderResource Relative path from test resources root to tracing level config directory
   */
  protected void configureTracingLevelSystemProperties(String configurationFolderResource) {
    this.tracingLevelConfigurationFileNameProperty =
        new SystemProperty(MULE_OPEN_TELEMETRY_TRACING_LEVEL_CONFIGURATION_FILE_NAME,
                           CONFIGURATION_FILE_NAME);
    this.tracingLevelConfigurationFilePathProperty =
        new SystemProperty(MULE_OPEN_TELEMETRY_TRACING_LEVEL_CONFIGURATION_FILE_PATH,
                           resolveAbsoluteConfigurationPath(configurationFolderResource));
  }

  /**
   * Resolves the absolute filesystem path for a tracing level configuration directory.
   *
   * @param tracingLevelResourcePath Relative path from test resources root
   * @return Absolute path to the configuration directory
   * @throws IllegalStateException if the resource cannot be found or resolved
   */
  protected String resolveAbsoluteConfigurationPath(String tracingLevelResourcePath) {
    URL resourceUrl = getClass().getClassLoader().getResource(tracingLevelResourcePath);
    if (resourceUrl == null) {
      throw new IllegalStateException(format("Configuration directory not found: %s", tracingLevelResourcePath));
    }
    try {
      return new File(resourceUrl.toURI()).getAbsolutePath();
    } catch (Exception e) {
      throw new IllegalStateException(format("Failed to resolve configuration path: %s", tracingLevelResourcePath), e);
    }
  }
}
