/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.test.components.tracing;

import static org.mule.runtime.module.observability.configuration.ObservabilityConfigurationFileWatcher.MULE_OBSERVABILITY_CONFIGURATION_WATCHER_DEFAULT_DELAY_PROPERTY;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.ALWAYS_ON_SAMPLER;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_OTEL_TRACES_SAMPLER;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_TRACING_CONFIGURATION_FILE_NAME;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_TRACING_CONFIGURATION_FILE_PATH;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.USE_MULE_OPEN_TELEMETRY_EXPORTER_SNIFFER;
import static org.mule.runtime.tracing.level.api.config.TracingLevel.MONITORING;
import static org.mule.runtime.tracing.level.api.config.TracingLevel.OVERVIEW;
import static org.mule.test.allure.AllureConstants.Profiling.PROFILING;
import static org.mule.test.allure.AllureConstants.Profiling.ProfilingServiceStory.DEFAULT_CORE_EVENT_TRACER;
import static org.mule.test.components.tracing.OpenTelemetryProtobufSpanUtils.getSpans;
import static org.mule.test.infrastructure.profiling.tracing.TracingTestUtils.createAttributeMap;
import static org.mule.test.infrastructure.profiling.tracing.TracingTestUtils.getDefaultAttributesToAssertExistence;

import static java.lang.System.clearProperty;
import static java.lang.System.currentTimeMillis;
import static java.lang.System.setProperty;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.nio.file.Files.write;

import static com.linecorp.armeria.common.HttpResponse.from;
import static com.linecorp.armeria.common.HttpStatus.OK;
import static com.linecorp.armeria.common.HttpStatus.REQUEST_TIMEOUT;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;

import org.mule.functional.junit4.MuleArtifactFunctionalTestCase;
import org.mule.runtime.tracer.api.sniffer.CapturedExportedSpan;
import org.mule.runtime.tracing.level.api.config.TracingLevel;
import org.mule.tck.probe.JUnitProbe;
import org.mule.tck.probe.PollingProber;
import org.mule.test.infrastructure.profiling.tracing.SpanTestHierarchy;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;

import com.linecorp.armeria.common.HttpRequest;
import com.linecorp.armeria.common.HttpResponse;
import com.linecorp.armeria.server.AbstractHttpService;
import com.linecorp.armeria.server.ServerBuilder;
import com.linecorp.armeria.server.ServiceRequestContext;
import com.linecorp.armeria.testing.junit4.server.ServerRule;
import io.opentelemetry.proto.collector.trace.v1.ExportTraceServiceRequest;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.jetbrains.annotations.NotNull;
import org.junit.After;
import org.junit.ClassRule;
import org.junit.Ignore;
import org.junit.Test;

/**
 * Integration test that verifies the ObservabilityConfigurationFileWatcher correctly detects and applies dynamic configuration
 * changes for both exporter and tracing level settings.
 */
@Feature(PROFILING)
@Story(DEFAULT_CORE_EVENT_TRACER)
@Ignore("To be completed after W-20062420")
public class ExportConfigurationChangeTestCase extends MuleArtifactFunctionalTestCase
    implements OpenTelemetryTracingTestRunnerConfigAnnotation {

  private static final String EXPECTED_FLOW_SPAN_NAME = "mule:flow";
  private static final String EXPECTED_SET_PAYLOAD_SPAN_NAME = "mule:set-payload";
  private static final String FLOW_LOCATION = "flow";
  private static final String SET_PAYLOAD_LOCATION = "flow/processors/0";
  private static final String TEST_ARTIFACT_ID = "ExportConfigurationChangeTestCase";

  private static final int TIMEOUT_MILLIS = 10000;
  private static final int POLL_DELAY_MILLIS = 100;
  private static final int MAX_BACKOFF_ATTEMPTS = 2;

  private File exporterConfigFile;
  private File tracingLevelFile;

  @ClassRule
  public static final TestServerRule server1 = new TestServerRule();

  @ClassRule
  public static final TestServerRule server2 = new TestServerRule();

  @Override
  protected String getConfigFile() {
    return "tracing/export-configuration-change.xml";
  }

  @Override
  protected void doSetUpBeforeMuleContextCreation() throws Exception {
    // Create exporter config file with configurable name in conf folder
    // Note: We use getConfFolder() because the tracing level configuration also reads from there
    // and we need both configurations to be accessible. FileTracingLevelConfiguration uses
    // ClassLoaderResourceProvider which requires files to be in the classpath (conf folder).
    File confFolder = getConfFolder();
    exporterConfigFile = new File(confFolder, "tracer-exporter-test.conf");
    writeExporterConfig(exporterConfigFile, false, server1.httpPort(), "HTTP");

    // Create tracing-level.conf in the conf folder to allow MuleContext initialization.
    // FileTracingLevelConfiguration requires this file with a fixed name in the conf folder.
    // We don't test tracing level reload in this test case (those tests are commented out below).
    tracingLevelFile = new File(confFolder, "tracing-level.conf");
    writeTracingLevelConfig(tracingLevelFile, MONITORING, null);

    // Set both the exporter config file name AND path to point to our test conf folder
    // FileSpanExporterConfiguration needs the absolute filesystem path for the file watcher
    // FileTracingLevelConfiguration should pick up "conf/" as a classpath resource
    setProperty(MULE_OPEN_TELEMETRY_TRACING_CONFIGURATION_FILE_NAME, exporterConfigFile.getName());
    setProperty(MULE_OPEN_TELEMETRY_TRACING_CONFIGURATION_FILE_PATH, confFolder.getAbsolutePath());
    System.out.println("DEBUG: Set MULE_OPEN_TELEMETRY_TRACING_CONFIGURATION_FILE_PATH to: " + confFolder.getAbsolutePath());

    // Use SimpleSpanProcessor (synchronous export) instead of BatchSpanProcessor (5s delay)
    // This allows the test to immediately observe exported spans without waiting for the batch delay
    setProperty(USE_MULE_OPEN_TELEMETRY_EXPORTER_SNIFFER, "true");

    setProperty(MULE_OPEN_TELEMETRY_OTEL_TRACES_SAMPLER, ALWAYS_ON_SAMPLER);
    setProperty(MULE_OBSERVABILITY_CONFIGURATION_WATCHER_DEFAULT_DELAY_PROPERTY, "100");
  }

  private File getConfFolder() {
    // Get the conf folder from the test classpath using the location of this test class
    // This ensures we get test-classes, not classes (main output)
    URL testClassUrl = getClass().getProtectionDomain().getCodeSource().getLocation();
    File testClasses = new File(testClassUrl.getPath());
    File confFolder = new File(testClasses, "conf");
    if (!confFolder.exists()) {
      confFolder.mkdirs();
    }
    System.out.println("DEBUG: Using conf folder: " + confFolder.getAbsolutePath());
    System.out.println("DEBUG: Conf folder exists: " + confFolder.exists());
    return confFolder;
  }

  @After
  public void doAfter() {
    clearProperty(MULE_OPEN_TELEMETRY_TRACING_CONFIGURATION_FILE_NAME);
    clearProperty(MULE_OPEN_TELEMETRY_TRACING_CONFIGURATION_FILE_PATH);
    clearProperty(USE_MULE_OPEN_TELEMETRY_EXPORTER_SNIFFER);
    clearProperty(MULE_OPEN_TELEMETRY_OTEL_TRACES_SAMPLER);
    clearProperty(MULE_OBSERVABILITY_CONFIGURATION_WATCHER_DEFAULT_DELAY_PROPERTY);
    server1.reset();
    server2.reset();

    // Clean up test config files from conf folder
    File confFolder = new File("target/test-classes/conf");
    if (confFolder.exists()) {
      new File(confFolder, "tracer-exporter-test.conf").delete();
      new File(confFolder, "tracing-level.conf").delete();
    }
  }

  // ============================================
  // EXPORTER CONFIG RELOAD TESTS
  // ============================================

  /**
   * Tests that the file watcher detects changes to the exporter enabled property and applies them dynamically without requiring a
   * restart.
   */
  @Test
  @Ignore("To be completed after W-20062420")
  public void testExporterEnabledDisabledReload() throws Exception {
    // Initial state: disabled, no spans should be exported
    flowRunner(FLOW_LOCATION).withPayload(TEST_PAYLOAD).run();
    assertThat(server1.getCapturedExportedSpans(), hasSize(0));

    // Change to enabled and wait for reload
    writeExporterConfig(exporterConfigFile, true, server1.httpPort(), "HTTP");
    waitForBehaviorChange(() -> {
      try {
        server1.reset();
        flowRunner(FLOW_LOCATION).withPayload(TEST_PAYLOAD).run();
        return server1.getCapturedExportedSpans().size() > 0;
      } catch (Exception e) {
        return false;
      }
    });
    assertThat(server1.getCapturedExportedSpans(), hasSize(2));
    assertExpectedSpanTreeMonitoring(server1.getCapturedExportedSpans());

    // Change back to disabled and wait for reload
    writeExporterConfig(exporterConfigFile, false, server1.httpPort(), "HTTP");
    waitForBehaviorChange(() -> {
      try {
        server1.reset();
        flowRunner(FLOW_LOCATION).withPayload(TEST_PAYLOAD).run();
        return server1.getCapturedExportedSpans().size() == 0;
      } catch (Exception e) {
        return false;
      }
    });

    assertThat(server1.getCapturedExportedSpans(), hasSize(0));
  }

  /**
   * Tests that the file watcher detects changes to the exporter endpoint property and applies them dynamically, routing spans to
   * the new endpoint.
   */
  @Test
  @Ignore("To be completed after W-20062420")
  public void testExporterEndpointReload() throws Exception {
    // Initial: server1
    writeExporterConfig(exporterConfigFile, true, server1.httpPort(), "HTTP");
    waitForBehaviorChange(() -> {
      try {
        server1.reset();
        flowRunner(FLOW_LOCATION).withPayload(TEST_PAYLOAD).run();
        return server1.getCapturedExportedSpans().size() > 0;
      } catch (Exception e) {
        return false;
      }
    });

    assertThat(server1.getCapturedExportedSpans(), hasSize(2));

    // Change to server2 and wait for reload
    writeExporterConfig(exporterConfigFile, true, server2.httpPort(), "HTTP");
    waitForBehaviorChange(() -> {
      try {
        server1.reset();
        server2.reset();
        flowRunner(FLOW_LOCATION).withPayload(TEST_PAYLOAD).run();
        return server2.getCapturedExportedSpans().size() > 0;
      } catch (Exception e) {
        return false;
      }
    });

    assertThat(server2.getCapturedExportedSpans(), hasSize(2));
    assertThat(server1.getCapturedExportedSpans(), hasSize(0));
  }

  // ============================================
  // TRACING LEVEL CONFIG RELOAD TESTS
  // ============================================

  /**
   * Tests that the file watcher detects changes to the tracing level property and applies them dynamically, changing the detail
   * level of captured spans.
   */
  @Test
  @Ignore("To be completed after W-20062420")
  public void testTracingLevelReload() throws Exception { // Enable exporter first
    writeExporterConfig(exporterConfigFile, true, server1.httpPort(), "HTTP");

    // Initial: MONITORING (2 spans: flow + set-payload)
    writeTracingLevelConfig(tracingLevelFile, MONITORING, null);
    waitForBehaviorChange(() -> {
      try {
        server1.reset();
        flowRunner(FLOW_LOCATION).withPayload(TEST_PAYLOAD).run();
        return server1.getCapturedExportedSpans().size() == 2;
      } catch (Exception e) {
        return false;
      }
    });
    assertThat(server1.getCapturedExportedSpans(), hasSize(2));
    assertExpectedSpanTreeMonitoring(server1.getCapturedExportedSpans());

    // Change to OVERVIEW (1 span: only flow)
    writeTracingLevelConfig(tracingLevelFile, OVERVIEW, null);
    waitForBehaviorChange(() -> {
      try {
        server1.reset();
        flowRunner(FLOW_LOCATION).withPayload(TEST_PAYLOAD).run();
        return server1.getCapturedExportedSpans().size() == 1;
      } catch (Exception e) {
        return false;
      }
    });

    assertThat(server1.getCapturedExportedSpans(), hasSize(1));
    assertExpectedSpanTreeOverview(server1.getCapturedExportedSpans());
  }

  /**
   * Tests that the file watcher detects changes to the tracing level overrides property and applies them dynamically, allowing
   * fine-grained control over specific flows.
   */
  @Test
  @Ignore("To be completed after W-20062420")
  public void testTracingLevelOverridesReload() throws Exception {
    writeExporterConfig(exporterConfigFile, true, server1.httpPort(), "HTTP");

    // Initial: OVERVIEW globally (1 span)
    writeTracingLevelConfig(tracingLevelFile, OVERVIEW, null);
    waitForBehaviorChange(() -> {
      try {
        server1.reset();
        flowRunner(FLOW_LOCATION).withPayload(TEST_PAYLOAD).run();
        return server1.getCapturedExportedSpans().size() == 1;
      } catch (Exception e) {
        return false;
      }
    });
    assertThat(server1.getCapturedExportedSpans(), hasSize(1));
    assertExpectedSpanTreeOverview(server1.getCapturedExportedSpans());

    // Add override: flow=MONITORING (2 spans)
    writeTracingLevelConfig(tracingLevelFile, OVERVIEW, List.of("flow=MONITORING"));
    waitForBehaviorChange(() -> {
      try {
        server1.reset();
        flowRunner(FLOW_LOCATION).withPayload(TEST_PAYLOAD).run();
        return server1.getCapturedExportedSpans().size() == 2;
      } catch (Exception e) {
        return false;
      }
    });

    assertThat(server1.getCapturedExportedSpans(), hasSize(2));
    assertExpectedSpanTreeMonitoring(server1.getCapturedExportedSpans());
  }


  // ============================================
  // HELPER METHODS
  // ============================================

  /**
   * Waits for a configuration reload to complete by polling until a specific behavior change is detected. This avoids race
   * conditions caused by arbitrary sleeps.
   *
   * @param condition A supplier that returns true when the expected behavior change has occurred
   */
  private void waitForBehaviorChange(Supplier<Boolean> condition) {
    new PollingProber(TIMEOUT_MILLIS, POLL_DELAY_MILLIS).check(new JUnitProbe() {

      @Override
      protected boolean test() {
        try {
          return condition.get();
        } catch (Exception e) {
          return false;
        }
      }

      @Override
      public String describeFailure() {
        return "Configuration reload was not detected within " + TIMEOUT_MILLIS + "ms";
      }
    });
  }

  /**
   * Writes an exporter configuration file in YAML format.
   */
  private void writeExporterConfig(File file, boolean enabled, int port, String type) throws IOException {
    String content = String.format(
                                   "mule:\n" +
                                       "  openTelemetry:\n" +
                                       "    tracer:\n" +
                                       "      exporter:\n" +
                                       "        enabled: %s\n" +
                                       "        type: %s\n" +
                                       "        endpoint: \"http://localhost:%d\"\n" +
                                       "        sampler: ALWAYS_ON\n",
                                   enabled, type, port);

    write(file.toPath(), content.getBytes(UTF_8));
    file.setLastModified(currentTimeMillis());
  }

  /**
   * Writes a tracing level configuration file in YAML format.
   *
   * @param file      The file to write to
   * @param level     The tracing level (MONITORING, OVERVIEW, DEBUG)
   * @param overrides Optional list of overrides (e.g., "flow=MONITORING")
   */
  private void writeTracingLevelConfig(File file, TracingLevel level, List<String> overrides) throws IOException {
    StringBuilder content = new StringBuilder();
    content.append("mule:\n");
    content.append("  openTelemetry:\n");
    content.append("    tracer:\n");
    content.append("      level: ").append(level.name()).append("\n");

    if (overrides != null && !overrides.isEmpty()) {
      content.append("      levelOverrides:\n");
      for (String override : overrides) {
        content.append("        - \"").append(override).append("\"\n");
      }
    }

    write(file.toPath(), content.toString().getBytes(UTF_8));
    file.setLastModified(currentTimeMillis());
  }

  /**
   * Asserts that the captured spans match the expected structure for OVERVIEW level.
   * <p>
   * OVERVIEW level only captures flow-level spans, so we expect: - Root span: "mule:flow" (the flow itself) - No child spans
   * (processors are not traced)
   * <p>
   * This method verifies the span hierarchy and attributes match OVERVIEW expectations.
   */
  private static void assertExpectedSpanTreeOverview(Collection<CapturedExportedSpan> exportedSpans) {
    List<String> attributesToAssertExistence = getDefaultAttributesToAssertExistence();

    SpanTestHierarchy expectedSpanHierarchy = new SpanTestHierarchy(exportedSpans);
    expectedSpanHierarchy.withRoot(EXPECTED_FLOW_SPAN_NAME)
        .addAttributesToAssertValue(createAttributeMap(FLOW_LOCATION, TEST_ARTIFACT_ID))
        .addAttributesToAssertExistence(attributesToAssertExistence);

    expectedSpanHierarchy.assertSpanTree();
  }

  /**
   * Asserts that the captured spans match the expected structure for MONITORING level.
   * <p>
   * MONITORING level captures both flow-level and component-level spans, so we expect: - Root span: "mule:flow" (the flow itself)
   * - Child span: "mule:set-payload" (the processor)
   * <p>
   * This method verifies not just the count, but also: - Span hierarchy (parent-child relationships) - Span names - Required
   * attributes (location, artifact ID, etc.)
   */
  private static void assertExpectedSpanTreeMonitoring(Collection<CapturedExportedSpan> exportedSpans) {
    List<String> attributesToAssertExistence = getDefaultAttributesToAssertExistence();
    Map<String, String> setPayloadAttributeMap = createAttributeMap(SET_PAYLOAD_LOCATION, TEST_ARTIFACT_ID);

    SpanTestHierarchy expectedSpanHierarchy = new SpanTestHierarchy(exportedSpans);
    expectedSpanHierarchy.withRoot(EXPECTED_FLOW_SPAN_NAME)
        .addAttributesToAssertValue(createAttributeMap(FLOW_LOCATION, TEST_ARTIFACT_ID))
        .addAttributesToAssertExistence(attributesToAssertExistence).beginChildren().child(EXPECTED_SET_PAYLOAD_SPAN_NAME)
        .addAttributesToAssertValue(setPayloadAttributeMap).addAttributesToAssertExistence(attributesToAssertExistence)
        .endChildren();

    expectedSpanHierarchy.assertSpanTree();
  }

  private static final class TestServerRule extends ServerRule {

    public static final String PATH_PATTERN = "/";

    private final List<CapturedExportedSpan> capturedExportedSpans = new ArrayList<>();

    private final AtomicInteger exportAttempts = new AtomicInteger(0);

    @Override
    protected void configure(ServerBuilder sb) {
      sb.service(PATH_PATTERN, new AbstractHttpService() {

        @Override
        protected @NotNull HttpResponse doPost(@NotNull ServiceRequestContext ctx, @NotNull HttpRequest req) {
          return HttpResponse.from(req.aggregate().handle((aReq, cause) -> {
            CompletableFuture<HttpResponse> responseFuture = new CompletableFuture<>();
            HttpResponse res = from(responseFuture);
            if (exportAttempts.incrementAndGet() < MAX_BACKOFF_ATTEMPTS) {
              responseFuture.complete(HttpResponse.of(REQUEST_TIMEOUT));
              return res;
            }
            try {
              capturedExportedSpans.addAll(
                                           getSpans(ExportTraceServiceRequest
                                               .parseFrom(new ByteArrayInputStream(aReq.content().array()))));
            } catch (IOException e) {
              // Nothing to do.
            }
            responseFuture.complete(HttpResponse.of(OK));
            return res;

          }));
        }
      });
      sb.http(0);
    }

    public List<CapturedExportedSpan> getCapturedExportedSpans() {
      return capturedExportedSpans;
    }

    public void reset() {
      capturedExportedSpans.clear();
    }
  }

}
