/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.test.integration.logging.otel;

import static org.mule.runtime.api.util.MuleSystemProperties.MULE_SIMPLE_LOG;
import static org.mule.runtime.api.util.IOUtils.getResourceAsUrl;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_ENABLED;
import static org.mule.runtime.logging.otel.impl.OpenTelemetryLoggingSDKFactory.getLogRecordSniffer;
import static org.mule.runtime.logging.otel.impl.export.log4j.OpenTelemetryLog4JBridge.OPEN_TELEMETRY_APPENDER_NAME_SUFFIX;
import static org.mule.tck.probe.PollingProber.probe;
import static org.mule.test.allure.AllureConstants.IntegrationTestsFeature.INTEGRATIONS_TESTS;
import static org.mule.test.allure.AllureConstants.Logging.LOGGING;
import static org.mule.test.allure.AllureConstants.Logging.LoggingStory.LOGGING_LIBS_SUPPORT;

import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;

import org.mule.runtime.logging.otel.impl.export.sniffer.ExportedLogRecordSniffer;
import org.mule.runtime.logging.otel.impl.export.sniffer.SniffedLogRecord;
import org.mule.runtime.module.deployment.impl.internal.builder.ApplicationFileBuilder;
import org.mule.runtime.module.deployment.impl.internal.builder.DomainFileBuilder;
import org.mule.runtime.test.integration.logging.AbstractLogConfigurationTestCase;
import org.mule.runtime.test.integration.logging.util.UseMuleLog4jContextFactory;
import org.mule.tck.junit4.rule.SystemProperty;

import org.junit.Rule;
import org.junit.Test;

import java.io.File;
import java.net.URL;
import java.nio.file.Files;
import java.util.List;

import io.qameta.allure.Feature;
import io.qameta.allure.Features;
import io.qameta.allure.Story;

@Features({@Feature(INTEGRATIONS_TESTS), @Feature(LOGGING)})
@Story(LOGGING_LIBS_SUPPORT)
public class Log4JBridgeConfigurationTestCase extends AbstractLogConfigurationTestCase {

  public static final String OTEL_LOGGING_APPLICATION = "otel-logging-application";
  public static final String OTEL_LOGGING_DOMAIN = "otel-logging-domain";

  private final ApplicationFileBuilder loggingAppFileBuilder =
      new ApplicationFileBuilder(OTEL_LOGGING_APPLICATION).definedBy("log/otel/otel-logging.xml")
          .usingResource("log/log4j-otel-config.xml",
                         "log4j2.xml");

  private final DomainFileBuilder loggingDomainFileBuilder =
      new DomainFileBuilder(OTEL_LOGGING_DOMAIN).definedBy("log/otel/otel-logging-domain.xml")
          .usingResource("log/log4j-otel-config.xml",
                         "log4j2.xml");

  private final ApplicationFileBuilder loggingAppWithWarnLevelFileBuilder =
      new ApplicationFileBuilder(OTEL_LOGGING_APPLICATION + "-warn").definedBy("log/otel/otel-logging-multiple-levels.xml")
          .usingResource("log/log4j-otel-config.xml", "log4j2.xml");

  private final ApplicationFileBuilder loggingAppWithDebug2LevelFileBuilder =
      new ApplicationFileBuilder(OTEL_LOGGING_APPLICATION + "-debug2").definedBy("log/otel/otel-logging-multiple-levels.xml")
          .usingResource("log/log4j-otel-config-debug.xml", "log4j2.xml");

  private final ApplicationFileBuilder loggingAppWithMultipleLevelsFileBuilder =
      new ApplicationFileBuilder(OTEL_LOGGING_APPLICATION + "-multiple-levels")
          .definedBy("log/otel/otel-logging-multiple-levels.xml")
          .usingResource("log/log4j-otel-config.xml", "log4j2.xml");

  @Rule
  public UseMuleLog4jContextFactory muleLogging = new UseMuleLog4jContextFactory();

  @Rule
  public SystemProperty enableOTELLogging = new SystemProperty(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_ENABLED, "true");

  @Rule
  public SystemProperty enableOTELLoggingSniffing =
      new SystemProperty(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER + ".use.sniffer", "true");

  @Override
  public void setUp() throws Exception {
    super.setUp();
    // here we're trying to test log separation so we need to
    // disable this default property of the fake mule server
    // in order to test that
    System.clearProperty(MULE_SIMPLE_LOG);
    muleServer.start();
  }

  private void copyLoggingExporterConfigToConf(String configResourceName) throws Exception {
    File confFolder = new File(muleServer.getMuleHome(), "conf");
    File configFile = new File(confFolder, "logging-exporter.conf");
    URL resourceUrl = getResourceAsUrl(configResourceName, this.getClass());
    Files.copy(resourceUrl.openStream(), configFile.toPath(), REPLACE_EXISTING);
  }

  @Test
  public void appenderConfigurationApp() throws Exception {
    muleServer.deploy(loggingAppFileBuilder.getArtifactFile().toUri().toURL(), OTEL_LOGGING_APPLICATION);
    assertThat(true, equalTo(loggerHasAppender(OTEL_LOGGING_APPLICATION, getRootLoggerForApp(OTEL_LOGGING_APPLICATION),
                                               OTEL_LOGGING_APPLICATION + OPEN_TELEMETRY_APPENDER_NAME_SUFFIX)));
  }

  @Test
  public void appenderConfigurationDomain() throws Exception {
    muleServer.deployDomainFile(loggingDomainFileBuilder.getArtifactFile().toFile());
    muleServer.deploy(loggingAppFileBuilder.getArtifactFile().toUri().toURL(), OTEL_LOGGING_APPLICATION);
    assertThat(true,
               equalTo(loggerHasAppender(OTEL_LOGGING_APPLICATION,
                                         getRootLoggerForDomain(OTEL_LOGGING_DOMAIN + "-1.0.0-mule-domain"),
                                         OTEL_LOGGING_APPLICATION + OPEN_TELEMETRY_APPENDER_NAME_SUFFIX)));
  }

  @Test
  public void appenderIsAdditiveAndLogsAllLevels() throws Exception {
    ExportedLogRecordSniffer exportedLogRecordSniffer = getLogRecordSniffer();
    muleServer.deploy(loggingAppFileBuilder.getArtifactFile().toUri().toURL(), OTEL_LOGGING_APPLICATION);
    probe(100000, 500, () -> exportedLogRecordSniffer.getSniffedLogRecords().stream()
        .anyMatch(sniffedLogRecord -> sniffedLogRecord.getBody().equals("Test")));
    SniffedLogRecord testSniffedLogRecord = exportedLogRecordSniffer.getSniffedLogRecords().stream()
        .filter(sniffedLogRecord -> sniffedLogRecord.getBody().equals("Test")).findFirst().get();
    assertThat(testSniffedLogRecord.getBody(), equalTo("Test"));
    assertThat(testSniffedLogRecord.getSeverity(), equalTo("INFO"));
  }

  @Test
  public void appenderFiltersLogsByDefaultLevel() throws Exception {
    // Test that when level is default (INFO), INFO and above are exported
    ExportedLogRecordSniffer exportedLogRecordSniffer = getLogRecordSniffer();
    muleServer.deploy(loggingAppWithMultipleLevelsFileBuilder.getArtifactFile().toUri().toURL(),
                      OTEL_LOGGING_APPLICATION + "-multiple-levels");

    // Wait for logs to be exported
    probe(100000, 500, () -> exportedLogRecordSniffer.getSniffedLogRecords().stream()
        .anyMatch(sniffedLogRecord -> sniffedLogRecord.getBody().equals("INFO message")));

    // Filter only the logs from our test application
    List<SniffedLogRecord> testLogs = exportedLogRecordSniffer.getSniffedLogRecords().stream()
        .filter(log -> log.getBody().equals("DEBUG message") || log.getBody().equals("INFO message")
            || log.getBody().equals("WARN message") || log.getBody().equals("ERROR message"))
        .toList();

    // Verify that INFO, WARN, and ERROR logs are exported
    assertThat("INFO log should be exported with default level",
               testLogs.stream().anyMatch(log -> log.getBody().equals("INFO message") && log.getSeverity().equals("INFO")),
               is(true));
    assertThat("WARN log should be exported with default level",
               testLogs.stream().anyMatch(log -> log.getBody().equals("WARN message") && log.getSeverity().equals("WARN")),
               is(true));
    assertThat("ERROR log should be exported with default level",
               testLogs.stream().anyMatch(log -> log.getBody().equals("ERROR message") && log.getSeverity().equals("ERROR")),
               is(true));

    // Verify that DEBUG log is NOT exported
    assertThat("DEBUG log should NOT be exported when level is INFO (default)",
               testLogs.stream().noneMatch(log -> log.getBody().equals("DEBUG message")),
               is(true));
  }

  @Test
  public void appenderFiltersLogsByConfiguredLevel() throws Exception {
    // Test that when level is configured as WARN, only WARN and above are exported
    copyLoggingExporterConfigToConf("log/otel/logging-exporter-warn.conf");

    String appName = OTEL_LOGGING_APPLICATION + "-warn";
    ExportedLogRecordSniffer exportedLogRecordSniffer = getLogRecordSniffer();
    muleServer.deploy(loggingAppWithWarnLevelFileBuilder.getArtifactFile().toUri().toURL(), appName);

    // Wait for logs to be exported
    probe(100000, 500, () -> exportedLogRecordSniffer.getSniffedLogRecords().stream()
        .anyMatch(sniffedLogRecord -> sniffedLogRecord.getBody().equals("WARN message")));

    // Filter only the logs from our test application
    List<SniffedLogRecord> testLogs = exportedLogRecordSniffer.getSniffedLogRecords().stream()
        .filter(log -> log.getBody().equals("DEBUG message") || log.getBody().equals("INFO message")
            || log.getBody().equals("WARN message") || log.getBody().equals("ERROR message"))
        .toList();

    // Verify that WARN and ERROR logs are exported
    assertThat("WARN log should be exported when level is WARN",
               testLogs.stream().anyMatch(log -> log.getBody().equals("WARN message") && log.getSeverity().equals("WARN")),
               is(true));
    assertThat("ERROR log should be exported when level is WARN",
               testLogs.stream().anyMatch(log -> log.getBody().equals("ERROR message") && log.getSeverity().equals("ERROR")),
               is(true));

    // Verify that DEBUG and INFO logs are NOT exported
    assertThat("DEBUG log should NOT be exported when level is WARN",
               testLogs.stream().noneMatch(log -> log.getBody().equals("DEBUG message")),
               is(true));
    assertThat("INFO log should NOT be exported when level is WARN",
               testLogs.stream().noneMatch(log -> log.getBody().equals("INFO message") && log.getSeverity().equals("INFO")),
               is(true));
  }

  @Test
  public void appenderFiltersLogsByDebug2Level() throws Exception {
    // Test that when level is configured as DEBUG2, it is normalized to DEBUG and all logs are exported
    copyLoggingExporterConfigToConf("log/otel/logging-exporter-debug2.conf");

    String appName = OTEL_LOGGING_APPLICATION + "-debug2";
    ExportedLogRecordSniffer exportedLogRecordSniffer = getLogRecordSniffer();
    muleServer.deploy(loggingAppWithDebug2LevelFileBuilder.getArtifactFile().toUri().toURL(), appName);

    // Wait for logs to be exported
    probe(100000, 500, () -> exportedLogRecordSniffer.getSniffedLogRecords().stream()
        .anyMatch(sniffedLogRecord -> sniffedLogRecord.getBody().equals("INFO message")));

    // Filter only the logs from our test application
    List<SniffedLogRecord> testLogs = exportedLogRecordSniffer.getSniffedLogRecords().stream()
        .filter(log -> log.getBody().equals("DEBUG message") || log.getBody().equals("INFO message")
            || log.getBody().equals("WARN message") || log.getBody().equals("ERROR message"))
        .toList();

    // Verify that all logs (DEBUG, INFO, WARN, ERROR) are exported when level is DEBUG2 (normalized to DEBUG)
    assertThat("DEBUG log should be exported when level is DEBUG2 (normalized to DEBUG)",
               testLogs.stream().anyMatch(log -> log.getBody().equals("DEBUG message") && log.getSeverity().equals("DEBUG")),
               is(true));
    assertThat("INFO log should be exported when level is DEBUG2 (normalized to DEBUG)",
               testLogs.stream().anyMatch(log -> log.getBody().equals("INFO message") && log.getSeverity().equals("INFO")),
               is(true));
    assertThat("WARN log should be exported when level is DEBUG2 (normalized to DEBUG)",
               testLogs.stream().anyMatch(log -> log.getBody().equals("WARN message") && log.getSeverity().equals("WARN")),
               is(true));
    assertThat("ERROR log should be exported when level is DEBUG2 (normalized to DEBUG)",
               testLogs.stream().anyMatch(log -> log.getBody().equals("ERROR message") && log.getSeverity().equals("ERROR")),
               is(true));
  }

}
