/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.test.integration.logging.otel;

import static org.mule.runtime.api.util.MuleSystemProperties.MULE_SIMPLE_LOG;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_ENABLED;
import static org.mule.runtime.logging.otel.impl.OpenTelemetryLoggingSDKFactory.getLogRecordSniffer;
import static org.mule.runtime.logging.otel.impl.export.log4j.OpenTelemetryLog4JBridge.OPEN_TELEMETRY_APPENDER_NAME_SUFFIX;
import static org.mule.tck.probe.PollingProber.probe;
import static org.mule.test.allure.AllureConstants.IntegrationTestsFeature.INTEGRATIONS_TESTS;
import static org.mule.test.allure.AllureConstants.Logging.LOGGING;
import static org.mule.test.allure.AllureConstants.Logging.LoggingStory.LOGGING_LIBS_SUPPORT;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.MatcherAssert.assertThat;

import org.mule.runtime.logging.otel.impl.export.sniffer.ExportedLogRecordSniffer;
import org.mule.runtime.logging.otel.impl.export.sniffer.SniffedLogRecord;
import org.mule.runtime.module.deployment.impl.internal.builder.ApplicationFileBuilder;
import org.mule.runtime.module.deployment.impl.internal.builder.DomainFileBuilder;
import org.mule.runtime.test.integration.logging.AbstractLogConfigurationTestCase;
import org.mule.runtime.test.integration.logging.util.UseMuleLog4jContextFactory;
import org.mule.tck.junit4.rule.SystemProperty;

import org.junit.Rule;
import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Features;
import io.qameta.allure.Story;

@Features({@Feature(INTEGRATIONS_TESTS), @Feature(LOGGING)})
@Story(LOGGING_LIBS_SUPPORT)
public class Log4JBridgeConfigurationTestCase extends AbstractLogConfigurationTestCase {

  public static final String OTEL_LOGGING_APPLICATION = "otel-logging-application";
  public static final String OTEL_LOGGING_DOMAIN = "otel-logging-domain";

  private final ApplicationFileBuilder loggingAppFileBuilder =
      new ApplicationFileBuilder(OTEL_LOGGING_APPLICATION).definedBy("log/otel/otel-logging.xml")
          .usingResource("log/log4j-otel-config.xml",
                         "log4j2.xml");

  private final DomainFileBuilder loggingDomainFileBuilder =
      new DomainFileBuilder(OTEL_LOGGING_DOMAIN).definedBy("log/otel/otel-logging-domain.xml")
          .usingResource("log/log4j-otel-config.xml",
                         "log4j2.xml");

  @Rule
  public UseMuleLog4jContextFactory muleLogging = new UseMuleLog4jContextFactory();

  @Rule
  public SystemProperty enableOTELLogging = new SystemProperty(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_ENABLED, "true");

  @Rule
  public SystemProperty enableOTELLoggingSniffing =
      new SystemProperty(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER + ".use.sniffer", "true");

  @Override
  public void setUp() throws Exception {
    super.setUp();
    // here we're trying to test log separation so we need to
    // disable this default property of the fake mule server
    // in order to test that
    System.clearProperty(MULE_SIMPLE_LOG);
    muleServer.start();
  }

  @Test
  public void appenderConfigurationApp() throws Exception {
    muleServer.deploy(loggingAppFileBuilder.getArtifactFile().toUri().toURL(), OTEL_LOGGING_APPLICATION);
    assertThat(true, equalTo(loggerHasAppender(OTEL_LOGGING_APPLICATION, getRootLoggerForApp(OTEL_LOGGING_APPLICATION),
                                               OTEL_LOGGING_APPLICATION + OPEN_TELEMETRY_APPENDER_NAME_SUFFIX)));
  }

  @Test
  public void appenderConfigurationDomain() throws Exception {
    muleServer.deployDomainFile(loggingDomainFileBuilder.getArtifactFile().toFile());
    muleServer.deploy(loggingAppFileBuilder.getArtifactFile().toUri().toURL(), OTEL_LOGGING_APPLICATION);
    assertThat(true,
               equalTo(loggerHasAppender(OTEL_LOGGING_APPLICATION,
                                         getRootLoggerForDomain(OTEL_LOGGING_DOMAIN + "-1.0.0-mule-domain"),
                                         OTEL_LOGGING_APPLICATION + OPEN_TELEMETRY_APPENDER_NAME_SUFFIX)));
  }

  @Test
  public void appenderIsAdditiveAndLogsAllLevels() throws Exception {
    ExportedLogRecordSniffer exportedLogRecordSniffer = getLogRecordSniffer();
    muleServer.deploy(loggingAppFileBuilder.getArtifactFile().toUri().toURL(), OTEL_LOGGING_APPLICATION);
    probe(100000, 500, () -> exportedLogRecordSniffer.getSniffedLogRecords().stream()
        .anyMatch(sniffedLogRecord -> sniffedLogRecord.getBody().equals("Test")));
    SniffedLogRecord testSniffedLogRecord = exportedLogRecordSniffer.getSniffedLogRecords().stream()
        .filter(sniffedLogRecord -> sniffedLogRecord.getBody().equals("Test")).findFirst().get();
    assertThat(testSniffedLogRecord.getBody(), equalTo("Test"));
    assertThat(testSniffedLogRecord.getSeverity(), equalTo("INFO"));
  }

}
