/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.shutdown;

import static org.mule.runtime.core.api.alert.MuleAlertingSupport.AlertNames.ALERT_EVENT_DISPATCHED_AFTER_STOP;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.startIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.stopIfNeeded;
import static org.mule.tck.probe.PollingProber.probe;
import static org.mule.test.allure.AllureConstants.LifecycleAndDependencyInjectionFeature.LIFECYCLE_AND_DEPENDENCY_INJECTION;
import static org.mule.test.allure.AllureConstants.LifecycleAndDependencyInjectionFeature.GracefulShutdownStory.GRACEFUL_SHUTDOWN_STORY;

import static java.util.Arrays.asList;
import static java.util.concurrent.Executors.newSingleThreadExecutor;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsMapContaining.hasKey;
import static org.hamcrest.core.IsIterableContaining.hasItem;

import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.core.api.construct.FlowConstruct;
import org.mule.test.allure.AllureConstants.SourcesFeature.SourcesStories;

import java.util.HashSet;

import org.junit.Test;

import io.qameta.allure.Feature;
import io.qameta.allure.Stories;
import io.qameta.allure.Story;
import jakarta.inject.Inject;
import jakarta.inject.Named;

@Feature(LIFECYCLE_AND_DEPENDENCY_INJECTION)
@Stories({
    @Story(SourcesStories.FLOW_DISPATCH),
    @Story(GRACEFUL_SHUTDOWN_STORY)
})
public class RogueExtensionShutdownTestCase extends AbstractShutdownTimeoutRequestResponseTestCase {

  @Inject
  @Named("neverStoppingSourceFlow")
  private FlowConstruct neverStoppingSourceFlow;

  @Inject
  @Named("stopFailsSourceFlow")
  private FlowConstruct stopFailsSourceFlow;

  @Inject
  @Named("stopHangsSourceFlow")
  private FlowConstruct stopHangsSourceFlow;

  @Override
  protected String getConfigFile() {
    return "org/mule/shutdown/rogue.xml";
  }

  @Test
  public void neverStoppingSource() throws Exception {
    startIfNeeded(neverStoppingSourceFlow);

    newSingleThreadExecutor().submit(() -> {
      try {
        stopIfNeeded(neverStoppingSourceFlow);
      } catch (MuleException e) {
        e.printStackTrace();
      }
    });

    probleForEventDispatchedAfterStopAlert("rogue:never-stopping-source", "neverStoppingSourceFlow");
  }

  @Test
  public void stopFailsSource() throws Exception {
    startIfNeeded(stopFailsSourceFlow);

    newSingleThreadExecutor().submit(() -> {
      try {
        stopIfNeeded(stopFailsSourceFlow);
      } catch (MuleException e) {
        e.printStackTrace();
      }
    });

    probleForEventDispatchedAfterStopAlert("rogue:stop-fails-source", "stopFailsSourceFlow");
  }

  @Test
  public void stopHangsSource() throws Exception {
    startIfNeeded(stopHangsSourceFlow);

    final var stopSubmit = newSingleThreadExecutor().submit(() -> {
      try {
        stopIfNeeded(stopHangsSourceFlow);
      } catch (MuleException e) {
        e.printStackTrace();
      }
    });

    probleForEventDispatchedAfterStopAlert("rogue:stop-hangs-source", "stopHangsSourceFlow");
    stopSubmit.cancel(true);
  }

  private void probleForEventDispatchedAfterStopAlert(final String sourceId, final String flowName) {
    probe(() -> {
      final var aggregatedAlerts = alertingSupport.alertsAggregation(() -> new HashSet<>(),
                                                                     (items, newItem) -> {
                                                                       items.add(newItem);
                                                                       return items;
                                                                     });

      assertThat(aggregatedAlerts, hasKey(ALERT_EVENT_DISPATCHED_AFTER_STOP));

      assertThat(aggregatedAlerts.get(ALERT_EVENT_DISPATCHED_AFTER_STOP).forLast1MinInterval(),
                 hasItem(asList(sourceId, flowName)));

      return true;
    });
  }

}
