/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.http.client;

import static org.mule.runtime.api.scheduler.SchedulerConfig.config;
import static org.mule.runtime.http.api.HttpConstants.Method.GET;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.api.util.concurrent.Latch;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.test.AbstractIntegrationTestCase;

import java.io.IOException;
import java.io.OutputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.RejectedExecutionException;

import io.qameta.allure.Issue;
import org.junit.After;
import org.junit.Rule;
import org.junit.Test;

// The reason we extend from AbstractIntegrationTestCase is to be able to use the real services, which are essential to reproduce
// the issue and verify the fix
@Issue("W-19806707")
public class HttpClientSchedulerBusyOnFutureCompletionTestCase extends AbstractIntegrationTestCase {

  private static final int SMALL_TIMEOUT_MS = 5000;

  @Rule
  public DynamicPort httpPort = new DynamicPort("httpPort");

  private HttpClient client;
  private TestServerThread server;

  @Override
  protected String getConfigFile() {
    // This is irrelevant, we need a config so the test doesn't fail to configure
    return "mule-config.xml";
  }

  @Override
  protected void doSetUp() throws Exception {
    client = getService(HttpService.class).getClientFactory().create(new HttpClientConfiguration.Builder()
        .setName("Test Client")
        .build());
    client.start();

    server = new TestServerThread(httpPort.getNumber());
    server.start();
  }

  @After
  public void stopClient() throws InterruptedException {
    if (client != null) {
      client.stop();
    }

    if (server != null) {
      server.join(SMALL_TIMEOUT_MS);
    }
  }

  @Test
  public void whenSchedulerBusyOnFutureCompletionThenDirectRuns() throws ExecutionException, InterruptedException {
    var releaseSchedulerLatch = new Latch();
    var schedulerBusyLatch = new Latch();
    var resultFuture = new CompletableFuture<HttpResponse>();

    // Creates a custom scheduler as target for the submitted task, we will want the backing pool of this scheduler to be busy
    // when submitting later, but we can't preemptively exhaust it because we need threads to hande the request and the response
    // Also we can't use a customScheduler with maxConcurrentTasks=1 because that would create a ByCallerThrottlingPolicy which
    // ignores the withDirectRunCpuLightWhenTargetBusy flag
    var otherScheduler = getService(SchedulerService.class).cpuLightScheduler();

    var response = client.sendAsync(HttpRequest.builder()
        .method(GET)
        .uri("http://localhost:%d/test".formatted(httpPort.getNumber()))
        .build());

    response.whenCompleteAsync((httpResponse, t) -> {
      // Just records the completion status so we can assert later on the main thread
      if (t != null) {
        resultFuture.completeExceptionally(t);
      } else {
        resultFuture.complete(httpResponse);
      }
    }, command -> {
      // This code block is executed asynchronously on a selector thread when the response future is completed

      // We wait until the backing scheduler is busy
      try {
        schedulerBusyLatch.await();
      } catch (InterruptedException e) {
        throw new RuntimeException(e);
      }

      // At this point there will be a rejected execution when submitting to this other scheduler which is busy
      // With the fix, the rejection policy will attempt to execute it in the same thread
      try {
        otherScheduler.submit(command);
      } catch (Throwable t) {
        // If there was any exception we will set it on the future so that it resurfaces on the main thread during the assertion
        resultFuture.completeExceptionally(t);
        throw t;
      }
    });

    // Make sure the future is completed (this does not wait for the whenCompleteAsync)
    response.get();

    // Now that the response has been received but before the whenComplete is executed, makes sure that the scheduler is busy
    ensureSchedulerBusy(otherScheduler, releaseSchedulerLatch, schedulerBusyLatch);

    // Here we expect a successful response, if the rejected execution was not handled we would see it raise at this point
    assertThat(resultFuture.get().getStatusCode(), is(200));

    releaseSchedulerLatch.release();
    otherScheduler.shutdown();
  }

  private void ensureSchedulerBusy(Scheduler scheduler, Latch releaseSchedulerLatch, Latch schedulerBusyLatch) {
    // Needs to make sure the thread doing the submissions is a scheduler thread otherwise it will use a WaitPolicy,
    // see ByCallerThreadGroupPolicy (it must also belong to a different thread group)
    getService(SchedulerService.class).customScheduler(config().withMaxConcurrentTasks(1)).submit(() -> {
      while (true) {
        try {
          scheduler.submit(() -> {
            releaseSchedulerLatch.await();
            return 0;
          });
        } catch (RejectedExecutionException e) {
          break;
        }
      }
      schedulerBusyLatch.release();
    });
  }

  private static class TestServerThread extends Thread {

    private final int port;

    private TestServerThread(int port) {
      this.port = port;
    }

    private Socket acceptOneConnection(int port) throws IOException {
      try (ServerSocket passiveSocket = new ServerSocket(port)) {
        return passiveSocket.accept();
      }
    }

    @Override
    public void run() {
      try (Socket peerSocket = acceptOneConnection(port)) {
        OutputStream outputStream = peerSocket.getOutputStream();
        outputStream.write(("""
            HTTP/1.1 200 OK\r
            Content-Length: 0\r
            \r
            """).getBytes());
        outputStream.flush();
        outputStream.close();
      } catch (IOException e) {
        throw new RuntimeException(e);
      }
    }
  }
}
