/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.test.integration.kerberos;

import static java.lang.System.clearProperty;
import static java.lang.System.getProperty;
import static java.lang.System.setProperty;

import static com.github.tomakehurst.wiremock.client.WireMock.configureFor;
import static org.slf4j.LoggerFactory.getLogger;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import org.apache.kerby.kerberos.kerb.server.SimpleKdcServer;
import com.github.tomakehurst.wiremock.WireMockServer;
import com.github.tomakehurst.wiremock.core.WireMockConfiguration;
import org.slf4j.Logger;

/**
 * Test infrastructure for Kerberos integration tests. Manages embedded KDC and WireMock servers.
 */
public class KerberosTestInfrastructure {

  private static final Logger LOGGER = getLogger(KerberosTestInfrastructure.class);

  // default values
  public static final String DEFAULT_TEST_REALM = "TESTREALM";
  public static final String DEFAULT_CLIENT_PRINCIPAL = "testuser";
  public static final String DEFAULT_SERVICE_HOST = "localhost";
  public static final String DEFAULT_FULL_CLIENT_PRINCIPAL = DEFAULT_CLIENT_PRINCIPAL + "@" + DEFAULT_TEST_REALM;

  private final String testRealm;
  private final String clientPrincipal;
  private final String serviceHost;
  private final String fullClientPrincipal;

  private SimpleKdcServer kdcServer;
  private WireMockServer mockServer;
  private File kdcWorkDir;
  private File clientKeytab;

  private String originalKrb5ConfProperty;
  private String originalJaasConfigProperty;

  /**
   * Default constructor using predefined constants for backward compatibility.
   */
  public KerberosTestInfrastructure() {
    this.testRealm = DEFAULT_TEST_REALM;
    this.clientPrincipal = DEFAULT_CLIENT_PRINCIPAL;
    this.serviceHost = DEFAULT_SERVICE_HOST;
    this.fullClientPrincipal = DEFAULT_FULL_CLIENT_PRINCIPAL;
  }


  public void start() throws Exception {
    setupKdcServer();
    setupMockServer();
    createKrb5ConfigFile();

    LOGGER.info("Kerberos infrastructure ready - KDC: {}, WireMock: {}",
                kdcServer.getKdcPort(), mockServer.port());
  }

  public void stop() throws Exception {
    if (mockServer != null && mockServer.isRunning()) {
      mockServer.stop();
    }

    if (kdcServer != null) {
      try {
        kdcServer.stop();
      } catch (Exception e) {
        LOGGER.warn("Error stopping KDC server: {}", e.getMessage());
      }
    }

    cleanupFiles();
    restoreSystemProperties();
  }

  private void setupKdcServer() throws Exception {
    kdcWorkDir = new File("target/kdc-work-dir");
    if (!kdcWorkDir.mkdirs() && !kdcWorkDir.exists()) {
      throw new RuntimeException("Failed to create KDC work directory: " + kdcWorkDir.getAbsolutePath());
    }

    kdcServer = new SimpleKdcServer();
    kdcServer.setWorkDir(kdcWorkDir);
    kdcServer.setKdcRealm(testRealm);
    kdcServer.setKdcHost(serviceHost);
    kdcServer.setAllowUdp(true);
    kdcServer.setAllowTcp(true);

    kdcServer.init();
    kdcServer.start();

    // create client principal
    kdcServer.createPrincipal(fullClientPrincipal, "password");

    // create service principal for HTTP service (proxy host:port)
    String proxyHost = serviceHost; // proxy runs on localhost

    // create service principal that matches what the client actually requests: HTTP/host/realm@REALM
    String fullServicePrincipal = "HTTP/" + proxyHost + "/" + testRealm.toLowerCase() + "@" + testRealm;
    kdcServer.createPrincipal(fullServicePrincipal, "service-password");

    // export service keytab (needed for service authentication)
    File serviceKeytab = new File(kdcWorkDir, "service.keytab");
    kdcServer.exportPrincipal(fullServicePrincipal, serviceKeytab);

    // export keytab for the client principal
    clientKeytab = new File(kdcWorkDir, "client.keytab");
    kdcServer.exportPrincipal(fullClientPrincipal, clientKeytab);

    LOGGER.info("KDC server started on port {}", kdcServer.getKdcPort());
  }

  private void setupMockServer() {
    mockServer = new WireMockServer(WireMockConfiguration.options()
        .port(0) // use random available port
        .asynchronousResponseEnabled(true)
        .asynchronousResponseThreads(10));

    mockServer.start();
    configureFor("localhost", mockServer.port());
  }

  private void createKrb5ConfigFile() throws IOException {
    // ensure target directory exists for configuration files
    File targetDir = new File("target");
    if (!targetDir.mkdirs() && !targetDir.exists()) {
      throw new RuntimeException("Failed to create target directory for configuration files");
    }

    File krb5ConfFile = new File("target/krb5.conf");
    try (FileWriter writer = new FileWriter(krb5ConfFile)) {
      writer.write("[libdefaults]\n");
      writer.write("    default_realm = " + testRealm + "\n");
      writer.write("    dns_lookup_kdc = false\n");
      writer.write("    dns_lookup_realm = false\n");
      writer.write("    ticket_lifetime = 24000\n");
      writer.write("    forwardable = yes\n");
      writer.write("    udp_preference_limit = 65536\n"); // Allow UDP first, then TCP fallback
      writer.write("\n");
      writer.write("[realms]\n");
      writer.write("    " + testRealm + " = {\n");
      writer.write("        kdc = " + serviceHost + ":" + kdcServer.getKdcPort() + "\n");
      writer.write("    }\n");
    }

    File jaasConfigFile = new File("target/jaas.conf");
    try (FileWriter writer = new FileWriter(jaasConfigFile)) {
      writer.write("KerberosClient {\n");
      writer.write("    com.sun.security.auth.module.Krb5LoginModule required\n");
      writer.write("    useKeyTab=true\n");
      writer.write("    keyTab=\"" + clientKeytab.getAbsolutePath() + "\"\n");
      writer.write("    principal=\"" + fullClientPrincipal + "\"\n");
      writer.write("    debug=true;\n");
      writer.write("};\n");
    }

    // save original properties for restoration
    originalKrb5ConfProperty = getProperty("java.security.krb5.conf");
    originalJaasConfigProperty = getProperty("java.security.auth.login.config");

    // set system properties
    setProperty("java.security.krb5.conf", krb5ConfFile.getAbsolutePath());
    setProperty("java.security.auth.login.config", jaasConfigFile.getAbsolutePath());

    LOGGER.debug("krb5.conf: {}", krb5ConfFile.getAbsolutePath());
    LOGGER.debug("jaas.conf: {}", jaasConfigFile.getAbsolutePath());
  }

  private void cleanupFiles() {
    if (clientKeytab != null && clientKeytab.exists()) {
      if (!clientKeytab.delete()) {
        LOGGER.warn("Failed to delete client keytab: {}", clientKeytab.getAbsolutePath());
      }
    }

    if (kdcWorkDir != null && kdcWorkDir.exists()) {
      deleteDirectoryRecursively(kdcWorkDir);
    }
  }

  private void deleteDirectoryRecursively(File dir) {
    File[] files = dir.listFiles();
    if (files != null) {
      for (File file : files) {
        if (file.isDirectory()) {
          deleteDirectoryRecursively(file);
        } else {
          if (!file.delete()) {
            LOGGER.warn("Failed to delete file: {}", file.getAbsolutePath());
          }
        }
      }
    }
    if (!dir.delete()) {
      LOGGER.warn("Failed to delete directory: {}", dir.getAbsolutePath());
    }
  }

  private void restoreSystemProperties() {
    restoreSystemProperty("java.security.krb5.conf", originalKrb5ConfProperty);
    restoreSystemProperty("java.security.auth.login.config", originalJaasConfigProperty);

    // Clear debug properties (if they were enabled)
    clearProperty("sun.security.krb5.debug");
    clearProperty("sun.security.jgss.debug");
  }

  private void restoreSystemProperty(String key, String originalValue) {
    if (originalValue != null) {
      setProperty(key, originalValue);
    } else {
      clearProperty(key);
    }
  }

  // Getters
  public SimpleKdcServer getKdcServer() {
    return kdcServer;
  }

  public WireMockServer getMockServer() {
    return mockServer;
  }

  public File getClientKeytab() {
    return clientKeytab;
  }

  public String getTestRealm() {
    return testRealm;
  }

  public String getClientPrincipal() {
    return clientPrincipal;
  }

  public String getServiceHost() {
    return serviceHost;
  }

  public String getFullClientPrincipal() {
    return fullClientPrincipal;
  }
}
