/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tls;

import static org.mule.test.allure.AllureConstants.TlsSsl.TLS_SSL_FEATURE;
import static org.mule.tls.TlsTestUtils.BOUNCY_CASTLE_CERTIFICATE_UNKNOWN_ERROR_MESSAGE;
import static org.mule.tls.TlsTestUtils.J11_SSL_ERROR_RESPONSE;
import static org.mule.tls.TlsTestUtils.J8_262_SSL_ERROR_RESPONSE;
import static org.mule.tls.fips.DefaultTestConfiguration.getDefaultEnvironmentConfiguration;

import static org.hamcrest.CoreMatchers.anyOf;
import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.instanceOf;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;

import static org.junit.internal.matchers.ThrowableMessageMatcher.hasMessage;

import org.mule.functional.api.exception.ExpectedError;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tck.junit4.rule.SystemProperty;
import org.mule.test.AbstractIntegrationTestCase;
import org.mule.test.runner.RunnerDelegateTo;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;

import org.junit.Rule;
import org.junit.Test;
import org.junit.runners.Parameterized;

import io.qameta.allure.Feature;

/**
 * Sets up two HTTPS clients using a regular trust-store, but one of them insecure. Then two HTTPS servers: one will return a
 * certificate present in the trust-store but with an invalid SAN extension, the other will return a certificate that's not in the
 * trust-store. Verifies that only the insecure client is successful.
 */
@Feature(TLS_SSL_FEATURE)
@RunnerDelegateTo(Parameterized.class)
public class HttpRequestTlsInsecureTestCase extends AbstractIntegrationTestCase {

  @Rule
  public SystemProperty serverKeyStore =
      new SystemProperty("serverKeyStore", getDefaultEnvironmentConfiguration().getTestServerKeyStore());

  @Rule
  public SystemProperty password = new SystemProperty("password", getDefaultEnvironmentConfiguration().getTestStorePassword());

  @Rule
  public SystemProperty storeType = new SystemProperty("storeType", getDefaultEnvironmentConfiguration().getTestStoreType());

  @Rule
  public SystemProperty sslCacerts = new SystemProperty("sslCacerts", getDefaultEnvironmentConfiguration().getTestSslCaCerts());

  @Rule
  public SystemProperty sslKeyStoreWithTestHostname =
      new SystemProperty("sslKeyStoreWithTestHostname", getDefaultEnvironmentConfiguration().getTestSslKeyStoreWithHostName());

  @Parameterized.Parameter
  public String config;

  @Rule
  public DynamicPort port1 = new DynamicPort("port1");
  @Rule
  public SystemProperty insecure = new SystemProperty("insecure", "true");
  @Rule
  public ExpectedError expectedError = ExpectedError.none();

  @Override
  protected String getConfigFile() {
    return config;
  }

  @Parameterized.Parameters
  public static Collection<Object[]> parameters() {
    return Arrays
        .asList(new Object[][] {{"http-request-insecure-hostname-config.xml"}, {"http-request-insecure-certificate-config.xml"}});
  }

  @Test
  public void insecureRequest() throws Exception {
    final CoreEvent res = flowRunner("testInsecureRequest").withPayload(TEST_PAYLOAD).run();
    assertThat(res.getMessage().getPayload().getValue(), is(TEST_PAYLOAD));
  }

  @Test
  public void secureRequest() throws Exception {
    expectedError.expectCause(instanceOf(IOException.class));
    expectedError
        .expectCause(anyOf(hasMessage(containsString(J8_262_SSL_ERROR_RESPONSE)),
                           hasMessage(containsString("No trusted certificate found")),
                           hasMessage(containsString(J11_SSL_ERROR_RESPONSE)),
                           hasMessage(containsString(BOUNCY_CASTLE_CERTIFICATE_UNKNOWN_ERROR_MESSAGE))));
    flowRunner("testSecureRequest").withPayload(TEST_PAYLOAD).run();
  }

}
