/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.functional.util.http;

import static java.util.Collections.list;

import static org.apache.commons.io.IOUtils.toByteArray;

import org.mule.runtime.api.util.MultiMap;

import java.io.IOException;
import java.util.Enumeration;

import jakarta.servlet.http.HttpServletRequest;

public class HttpMessage {

  private byte[] body = new byte[0];
  private String uri;
  private MultiMap<String, String> queryParams;
  private MultiMap<String, String> headers = new MultiMap<>();

  public HttpMessage(HttpServletRequest request) throws IOException {
    this.body = toByteArray(request.getInputStream());
    MultiMap<String, String> headersBuilder = new MultiMap<>();
    Enumeration<String> headerNames = request.getHeaderNames();
    while (headerNames.hasMoreElements()) {
      String headerName = headerNames.nextElement();
      headersBuilder.put(headerName, list(request.getHeaders(headerName)));
    }
    this.headers = headersBuilder;
    uri = request.getRequestURI();
    queryParams = queryToMap(request.getQueryString());
  }

  public byte[] getBody() {
    return body;
  }

  public MultiMap<String, String> getHeaders() {
    return headers;
  }


  public MultiMap<String, String> getQueryParams() {
    return queryParams;
  }

  public String getUri() {
    return uri;
  }

  private MultiMap<String, String> queryToMap(String query) {
    MultiMap<String, String> result = new MultiMap<>();
    if (query == null) {
      return result;
    }
    for (String param : query.split("&")) {
      String pair[] = param.split("=");
      if (pair.length > 1) {
        result.put(pair[0], pair[1]);
      } else {
        result.put(pair[0], "");
      }
    }
    return result;
  }
}
