/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.maven.client.maven.context.it;

import static org.mule.maven.client.api.MavenClientProvider.discoverProvider;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;

import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.MavenClientProvider;
import org.mule.maven.client.api.SettingsSupplierFactory;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.pom.parser.api.model.BundleDependency;
import org.mule.maven.pom.parser.api.model.BundleDescriptor;

import java.io.File;
import java.util.Optional;

import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.artifact.DefaultArtifact;

public class DependencyResolver implements AutoCloseable {

  private static final MavenClientProvider mavenClientProvider =
      discoverProvider(DependencyResolver.class.getClassLoader());

  private final MavenClient mavenClient;

  /**
   * Create a new instance with a default Maven configuration.
   */
  public DependencyResolver() {
    this(getDefaultMavenConfiguration());
  }

  /**
   * Create a new instance with a default Maven configuration with the given local repository.
   */
  public DependencyResolver(String localRepository) {
    this(getDefaultMavenConfiguration(localRepository));
  }

  /**
   * Create a new instance with the provided Maven configuration.
   */
  public DependencyResolver(MavenConfiguration mavenConfig) {
    mavenClient = mavenClientProvider.createMavenClient(mavenConfig);
  }

  private static MavenConfiguration getDefaultMavenConfiguration() {
    return getDefaultMavenConfiguration(null);
  }

  private static MavenConfiguration getDefaultMavenConfiguration(String localRepository) {
    final SettingsSupplierFactory settingsSupplierFactory = mavenClientProvider.getSettingsSupplierFactory();

    final Optional<File> globalSettings = settingsSupplierFactory.environmentGlobalSettingsSupplier();
    final Optional<File> userSettings = settingsSupplierFactory.environmentUserSettingsSupplier();
    final Optional<File> settingsSecurity = settingsSupplierFactory.environmentSettingsSecuritySupplier();

    final File localMavenRepository = localRepository != null ? new File(localRepository)
        : mavenClientProvider.getLocalRepositorySuppliers().environmentMavenRepositorySupplier().get();

    final MavenConfiguration.MavenConfigurationBuilder mavenConfigurationBuilder = newMavenConfigurationBuilder()
        .forcePolicyUpdateNever(true)
        .localMavenRepositoryLocation(localMavenRepository);

    globalSettings.ifPresent(mavenConfigurationBuilder::globalSettingsLocation);
    userSettings.ifPresent(mavenConfigurationBuilder::userSettingsLocation);
    settingsSecurity.ifPresent(mavenConfigurationBuilder::settingsSecurityLocation);

    return mavenConfigurationBuilder.build();
  }

  public BundleDependency resolveFor(String artifactStr) {
    Artifact artifact = new DefaultArtifact(artifactStr);
    BundleDescriptor mavenBundleDescriptor =
        new org.mule.maven.pom.parser.api.model.BundleDescriptor.Builder()
            .setGroupId(artifact.getGroupId())
            .setArtifactId(artifact.getArtifactId())
            .setVersion(artifact.getVersion())
            .setBaseVersion(artifact.getBaseVersion())
            .setType(artifact.getExtension())
            .setClassifier(artifact.getClassifier())
            .build();

    return mavenClient.resolveBundleDescriptor(mavenBundleDescriptor);
  }

  @Override
  public void close() throws Exception {
    mavenClient.close();
  }
}
