/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.server;

import static org.mule.runtime.http.api.HttpConstants.HttpStatus.INTERNAL_SERVER_ERROR;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.service.http.test.netty.utils.TestUtils.measuringNanoseconds;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.concurrent.TimeUnit.MILLISECONDS;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalToCompressingWhiteSpace;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.lessThan;

import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.service.http.netty.impl.message.content.StringHttpEntity;
import org.mule.service.http.test.common.server.AbstractHttpServerTestCase;
import org.mule.service.http.test.netty.utils.NoOpResponseStatusCallback;
import org.mule.service.http.test.netty.utils.ResponseWithoutHeaders;
import org.mule.service.http.test.netty.utils.TcpTextClient;

import java.io.IOException;

import io.qameta.allure.Issue;
import org.apache.commons.io.IOUtils;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

@Issue("W-17464403")
class ServerConnectionTimeoutTestCase extends AbstractHttpServerTestCase {

  // We want the timeouts to be caused by connection timeout, not by read timeout.
  private static final int SMALL_CONNECTION_TIMEOUT_MILLIS = 500;
  private static final long LARGE_READ_TIMEOUT_MILLIS = 1000000L;

  public ServerConnectionTimeoutTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @Override
  protected String getServerName() {
    return "test-server";
  }

  @Override
  protected HttpServerConfiguration.Builder configureServer(HttpServerConfiguration.Builder builder) {
    return builder
        .setConnectionIdleTimeout(SMALL_CONNECTION_TIMEOUT_MILLIS)
        .setReadTimeout(LARGE_READ_TIMEOUT_MILLIS);
  }

  @BeforeEach
  void setup() throws Exception {
    setUpServer();
    server.addRequestHandler("/test", (requestContext, responseCallback) -> {
      try {
        var asString = IOUtils.toString(requestContext.getRequest().getEntity().getContent(), UTF_8);
        responseCallback.responseReady(new ResponseWithoutHeaders(OK, new StringHttpEntity(asString)),
                                       new NoOpResponseStatusCallback());
      } catch (IOException e) {
        responseCallback.responseReady(new ResponseWithoutHeaders(INTERNAL_SERVER_ERROR, new StringHttpEntity(e.toString())),
                                       new NoOpResponseStatusCallback());
      }
    }).start();
  }

  @Test
  void sendPartialRequestShouldTimeout() throws IOException {
    try (TcpTextClient tcpTextClient = new TcpTextClient("localhost", port)) {
      // Send one request...
      tcpTextClient.sendString("""
          GET /test HTTP/1.1
          Host: localhost: %s

          """.formatted(port));

      // The response should be received normally.
      assertThat(tcpTextClient.receiveUntil("\r\n\r\n"), equalToCompressingWhiteSpace("""
          HTTP/1.1 200 OK
          content-length: 0"""));

      // Send second request (persistent connection)
      tcpTextClient.sendString("""
          GET /test HTTP/1.1
          Host: localhost: %d

          """.formatted(port));

      // The response should be received normally again.
      assertThat(tcpTextClient.receiveUntil("\r\n\r\n"), equalToCompressingWhiteSpace("""
          HTTP/1.1 200 OK
          content-length: 0"""));

      // Connection will be closed by connection timeout.
      long elapsedNanos = measuringNanoseconds(() -> {
        String content = tcpTextClient.receiveUntil("\r\n\r\n");
        assertThat(content, is(""));
      });

      long toleranceNanos = MILLISECONDS.toNanos(50L);
      long connectionTimeoutNanos = MILLISECONDS.toNanos(SMALL_CONNECTION_TIMEOUT_MILLIS);
      assertThat(elapsedNanos, is(lessThan(connectionTimeoutNanos + toleranceNanos)));
    }
  }
}
