/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static java.util.Objects.requireNonNull;

import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.async.ResponseStatusCallback;
import org.mule.service.http.netty.impl.streaming.ResponseStatusCallbackAdapter;

import java.io.IOException;
import java.io.InputStream;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandlerContext;

/**
 * Simplest implementation of a {@link BaseResponseSender}, which just sends the content by converting the
 * {@link org.mule.runtime.http.api.domain.entity.HttpEntity}'s content to a {@link ByteBuf}.
 */
public class DirectResponseSender extends BaseResponseSender {

  private final HttpResponse response;

  public DirectResponseSender(HttpRequest httpRequest, ChannelHandlerContext ctx, HttpResponse response,
                              ResponseStatusCallback statusCallback, HttpWriter writer) {
    super(httpRequest, ctx, response, new ResponseStatusCallbackAdapter(statusCallback), writer);
    requireNonNull(response.getEntity().getContent(), "Response content cannot be null");
    this.response = response;
  }

  @Override
  protected void sendContent() throws IOException {
    InputStream contentAsInputStream = response.getEntity().getContent();
    ByteBuf content = inputStreamToByteBuf(contentAsInputStream);
    sendLastContentAndFinishStreaming(content, contentAsInputStream);
  }

  private ByteBuf inputStreamToByteBuf(InputStream inputStream) throws IOException {
    int availableBytes = inputStream.available();
    ByteBuf content = createBuffer(availableBytes);
    content.writeBytes(inputStream, availableBytes);
    return content;
  }
}
