/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server.sse;

import static io.netty.handler.codec.http.LastHttpContent.EMPTY_LAST_CONTENT;

import org.mule.runtime.http.api.server.async.ResponseStatusCallback;
import org.mule.service.http.netty.impl.server.NettyHttp1ResponseReadyCallback;

import java.io.IOException;
import java.io.Writer;
import java.nio.charset.Charset;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.http.DefaultHttpContent;

/**
 * Implementation of {@link Writer} returned by {@link NettyHttp1ResponseReadyCallback#startResponse} so the caller can write the
 * response body. On {@link #write}, it accumulates the written data in a {@link StringBuilder}, and on {@link #flush()}, it sends
 * the data to the Netty's {@link ChannelHandlerContext ctx}. On {@link #close()}, it writes an empty "last" content to Netty, and
 * marks the {@link ResponseStatusCallback callback} as completed successfully.
 */
public class ResponseBodyWriter extends Writer {

  private final ChannelHandlerContext ctx;
  private final Charset encoding;
  private final ResponseStatusCallback callback;
  private final StringBuilder stringBuilder;

  public ResponseBodyWriter(ChannelHandlerContext ctx, Charset encoding, ResponseStatusCallback callback) {
    this.ctx = ctx;
    this.encoding = encoding;
    this.callback = callback;
    this.stringBuilder = new StringBuilder();
  }

  @Override
  public void write(char[] charBuffer, int off, int len) throws IOException {
    stringBuilder.append(charBuffer, off, len);
  }

  @Override
  public void flush() {
    byte[] bytes = stringBuilder.toString().getBytes(encoding);
    ByteBuf byteBuf = ctx.alloc().buffer(bytes.length);
    byteBuf.writeBytes(bytes);
    stringBuilder.setLength(0);
    ctx.writeAndFlush(new DefaultHttpContent(byteBuf));
  }

  @Override
  public void close() throws IOException {
    ctx.writeAndFlush(EMPTY_LAST_CONTENT);
    callback.responseSendSuccessfully();
  }
}
