/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.message;

import static io.netty.handler.codec.http.HttpResponseStatus.OK;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.when;

import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.service.http.netty.impl.message.ReactorNettyResponseWrapper;

import java.util.List;

import io.netty.handler.codec.http.DefaultHttpHeaders;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import reactor.netty.http.client.HttpClientResponse;

@ExtendWith(MockitoExtension.class)
class ReactorNettyResponseWrapperTestCase {

  @Mock
  private HttpClientResponse reactorNettyResponse;

  @Mock
  private HttpEntity body;

  @Test
  void status() {
    setUpStatus();
    var muleResponse = new ReactorNettyResponseWrapper(reactorNettyResponse, body);
    assertThat(muleResponse.getStatusCode(), is(200));
    assertThat(muleResponse.getReasonPhrase(), is("OK"));
  }

  @Test
  void headersIndividually() {
    setUpHeaders();
    var muleResponse = new ReactorNettyResponseWrapper(reactorNettyResponse, body);

    assertThat(muleResponse.getHeaderValue("content-length"), is("0"));
    assertThat(muleResponse.getHeaderValueIgnoreCase("cOnTeNt-leNGth"), is("0"));

    assertThat(muleResponse.getHeaderValue("X-Header1"), is("value"));
    assertThat(muleResponse.getHeaderValueIgnoreCase("x-HEADER1"), is("value"));

    assertThat(muleResponse.getHeaderValues("x-header2"), contains("value1", "value2"));
    assertThat(muleResponse.getHeaderValuesIgnoreCase("x-hEADer2"), contains("value1", "value2"));
  }

  @Test
  void headerNames() {
    setUpHeaders();
    var muleResponse = new ReactorNettyResponseWrapper(reactorNettyResponse, body);
    assertThat(muleResponse.getHeaderNames(), contains("Content-Length", "X-Header1", "X-Header2"));
  }

  @Test
  void allHeaders() {
    setUpHeaders();
    var muleResponse = new ReactorNettyResponseWrapper(reactorNettyResponse, body);
    var headers = muleResponse.getHeaders();

    assertThat(headers.get("content-length"), is("0"));
    assertThat(headers.get("X-Header1"), is("value"));
    assertThat(headers.getAll("x-header2"), contains("value1", "value2"));
  }

  @Test
  void entity() {
    var muleResponse = new ReactorNettyResponseWrapper(reactorNettyResponse, body);
    assertThat(muleResponse.getEntity(), is(body));
  }

  void setUpStatus() {
    when(reactorNettyResponse.status()).thenReturn(OK);
  }

  void setUpHeaders() {
    var headers = new DefaultHttpHeaders();
    headers.add("Content-Length", "0");
    headers.add("X-Header1", "value");
    headers.add("X-Header2", List.of("value1", "value2"));
    when(reactorNettyResponse.responseHeaders()).thenReturn(headers);
  }
}
