/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.sse;

import static org.mule.service.http.test.common.util.HttpMessageHeaderMatcher.hasHeader;
import static org.mule.service.http.test.netty.AllureConstants.SSE;
import static org.mule.service.http.test.netty.AllureConstants.SseStory.SSE_ENDPOINT;
import static org.mule.service.http.test.netty.AllureConstants.SseStory.SSE_SOURCE;
import static org.mule.service.http.test.netty.utils.sse.ServerSentEventTypeSafeMatcher.aServerSentEvent;

import static java.lang.String.format;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThrows;
import static org.junit.Assert.fail;
import static org.junit.internal.matchers.ThrowableMessageMatcher.hasMessage;

import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.runtime.http.api.server.ServerCreationException;
import org.mule.runtime.http.api.sse.client.SseRetryConfig;
import org.mule.runtime.http.api.sse.client.SseSource;
import org.mule.runtime.http.api.sse.client.SseSourceConfig;
import org.mule.service.http.test.common.AbstractHttpServiceTestCase;
import org.mule.service.http.test.netty.utils.sse.SSEEventsAggregator;
import org.mule.tck.junit5.DynamicPort;

import java.util.concurrent.ExecutionException;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

/**
 * Test for the communication between the SSE Source (client-side), and the SSE Endpoint (server-side).
 */
@Feature(SSE)
@Story(SSE_SOURCE)
@Story(SSE_ENDPOINT)
public class SseTestCase extends AbstractHttpServiceTestCase {

  private static final SseRetryConfig DONT_RETRY_ON_EOS = new SseRetryConfig(true, 2000L, false);

  @DynamicPort(systemProperty = "serverPort")
  Integer serverPort;

  private String sseUrl;

  private HttpServer httpServer;

  private HttpClient httpClient;
  private SseSourceConfig sseConfig;

  public SseTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @BeforeEach
  void setUp() throws Exception {
    sseUrl = format("http://localhost:%d/sse", serverPort);
    sseConfig = SseSourceConfig.builder(sseUrl)
        .withRetryConfig(DONT_RETRY_ON_EOS)
        .build();
    httpServer = getHttpServer(serverPort);
    httpServer.start();
    httpServer.sse("/sse",
                   ctx -> ctx.customizeResponse(res -> res.addResponseHeader("X-My-Custom-Header", "Value")),
                   sseClient -> {
                     try (sseClient) {
                       for (int i = 0; i < 10; ++i) {
                         sseClient.sendEvent("first", format("Event %d", i), "asd");
                         sseClient.sendEvent("second", format("Event %d", i));
                       }
                     } catch (Exception e) {
                       fail(e.getMessage());
                     }
                   });

    httpClient = service.getClientFactory().create(new HttpClientConfiguration.Builder()
        .setName("SSE Client")
        .setStreaming(true)
        .build());
    httpClient.start();
  }

  @AfterEach
  void tearDown() {
    httpClient.stop();
    httpServer.stop().dispose();
  }

  @Test
  void sseSourceNeedsTheClientStreamingEnabled() {
    HttpClient nonStreamingClient = service.getClientFactory().create(new HttpClientConfiguration.Builder()
        .setName("No-Streaming Client")
        .setStreaming(false)
        .build());
    var exception = assertThrows(IllegalStateException.class, () -> nonStreamingClient.sseSource(sseConfig));
    assertThat(exception, hasMessage(is("SSE source requires streaming enabled for client 'No-Streaming Client'")));
  }

  @Test
  void allEventsToFallbackListener() throws InterruptedException, ExecutionException {
    SSEEventsAggregator fallbackListener = new SSEEventsAggregator();

    SseSource sse = httpClient.sseSource(sseConfig);
    sse.register(fallbackListener);
    sse.open();

    assertThat(fallbackListener.getList(), contains(
                                                    aServerSentEvent("first", "Event 0"),
                                                    aServerSentEvent("second", "Event 0"),
                                                    aServerSentEvent("first", "Event 1"),
                                                    aServerSentEvent("second", "Event 1"),
                                                    aServerSentEvent("first", "Event 2"),
                                                    aServerSentEvent("second", "Event 2"),
                                                    aServerSentEvent("first", "Event 3"),
                                                    aServerSentEvent("second", "Event 3"),
                                                    aServerSentEvent("first", "Event 4"),
                                                    aServerSentEvent("second", "Event 4"),
                                                    aServerSentEvent("first", "Event 5"),
                                                    aServerSentEvent("second", "Event 5"),
                                                    aServerSentEvent("first", "Event 6"),
                                                    aServerSentEvent("second", "Event 6"),
                                                    aServerSentEvent("first", "Event 7"),
                                                    aServerSentEvent("second", "Event 7"),
                                                    aServerSentEvent("first", "Event 8"),
                                                    aServerSentEvent("second", "Event 8"),
                                                    aServerSentEvent("first", "Event 9"),
                                                    aServerSentEvent("second", "Event 9")));
  }

  @Test
  void multiplexEventsByName() throws InterruptedException, ExecutionException {
    SSEEventsAggregator firstListener = new SSEEventsAggregator();
    SSEEventsAggregator fallbackListener = new SSEEventsAggregator();

    SseSource sse = httpClient.sseSource(sseConfig);
    sse.register("first", firstListener);
    sse.register(fallbackListener);
    sse.open();

    assertThat(firstListener.getList(), contains(
                                                 aServerSentEvent("first", "Event 0"),
                                                 aServerSentEvent("first", "Event 1"),
                                                 aServerSentEvent("first", "Event 2"),
                                                 aServerSentEvent("first", "Event 3"),
                                                 aServerSentEvent("first", "Event 4"),
                                                 aServerSentEvent("first", "Event 5"),
                                                 aServerSentEvent("first", "Event 6"),
                                                 aServerSentEvent("first", "Event 7"),
                                                 aServerSentEvent("first", "Event 8"),
                                                 aServerSentEvent("first", "Event 9")));

    assertThat(fallbackListener.getList(), contains(
                                                    aServerSentEvent("second", "Event 0"),
                                                    aServerSentEvent("second", "Event 1"),
                                                    aServerSentEvent("second", "Event 2"),
                                                    aServerSentEvent("second", "Event 3"),
                                                    aServerSentEvent("second", "Event 4"),
                                                    aServerSentEvent("second", "Event 5"),
                                                    aServerSentEvent("second", "Event 6"),
                                                    aServerSentEvent("second", "Event 7"),
                                                    aServerSentEvent("second", "Event 8"),
                                                    aServerSentEvent("second", "Event 9")));
  }

  @Test
  @Issue("W-18599423")
  void serverSendsCustomHeader() throws ExecutionException, InterruptedException {
    HttpRequest request = HttpRequest.builder()
        .uri(sseUrl)
        .addHeader("Accept", "text/event-stream")
        .build();

    var future = httpClient.sendAsync(request);

    HttpResponse response = future.get();
    assertThat(response, hasHeader("X-My-Custom-Header", "Value"));
  }

  private HttpServer getHttpServer(int port) throws ServerCreationException {
    return service.getServerFactory().create(new HttpServerConfiguration.Builder()
        .setName("SSE Server")
        .setHost("localhost")
        .setPort(port)
        .build());
  }
}
