/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.streaming;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.LinkedList;
import java.util.Queue;

/**
 * Bidirectional implementation of a stream of bytes, which allows both writing to and read from the buffer.
 */
public class BidirectionalByteBufferStream {

  private final Queue<ByteArrayInputStream> writtenData;

  private int length;

  public BidirectionalByteBufferStream() {
    this.writtenData = new LinkedList<>();
    this.length = 0;
  }

  /**
   * Writes <code>len</code> bytes from the specified byte array starting at offset <code>off</code> to this output stream.
   *
   * @param buf the data.
   * @param off the start offset in the data.
   * @param len the number of bytes to write.
   *
   * @see OutputStream#write(byte[], int, int)
   */
  public void write(byte[] buf, int off, int len) {
    this.writtenData.add(new ByteArrayInputStream(buf, off, len));
    this.length += len;
  }

  /**
   * Retrieves the requested amount of bytes from the stream. If no data is present in the buffer, it returns -1.
   *
   * @param buf The buffer where the (output) bytes will be stored.
   * @param off The start offset in array <code>b</code> at which the data is written.
   * @param len the maximum number of bytes to read.
   *
   * @return the total number of bytes read into the buffer, or <code>-1</code> if there is no more data and the structure was
   *         closed.
   * @throws IOException if an error happens during the read.
   *
   * @see InputStream#read(byte[], int, int)
   */
  public int read(byte[] buf, int off, int len) throws IOException {
    if (this.length == 0 || len == 0) {
      return 0;
    }

    ByteArrayInputStream head = writtenData.peek();
    if (head == null) {
      return 0;
    }

    int bytesCurrentlyRead = 0;
    while (bytesCurrentlyRead < len && this.length > 0) {
      int readInThisIteration = readFromHead(buf, off + bytesCurrentlyRead, len - bytesCurrentlyRead);
      if (readInThisIteration == 0 || readInThisIteration == -1) {
        break;
      }
      bytesCurrentlyRead += readInThisIteration;
    }

    return bytesCurrentlyRead;
  }

  /**
   * @return <code>true</code> if the buffer is empty, or <code>false</code> otherwise.
   */
  public boolean isEmpty() {
    return length == 0;
  }

  private int readFromHead(byte[] buf, int off, int len) throws IOException {
    ByteArrayInputStream head = writtenData.peek();
    if (head == null) {
      return 0;
    }

    int result = head.read(buf, off, len);
    this.length -= result;

    if (head.available() == 0) {
      // No more data in this chunk.
      head.close();
      writtenData.remove();
    }

    return result;
  }
}
