/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server.util;

import static org.mule.runtime.api.metadata.MediaType.TEXT;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;

import static java.lang.String.format;

import static org.apache.commons.lang3.StringEscapeUtils.escapeHtml4;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.domain.request.HttpRequestContext;
import org.mule.runtime.http.api.server.RequestHandler;
import org.mule.runtime.http.api.server.async.HttpResponseReadyCallback;
import org.mule.runtime.http.api.server.async.ResponseStatusCallback;

import java.io.ByteArrayInputStream;

import org.slf4j.Logger;

public class ErrorRequestHandler implements RequestHandler {

  private static final Logger LOGGER = getLogger(ErrorRequestHandler.class);

  private final int statusCode;
  private final String reasonPhrase;
  protected String entityFormat;

  public ErrorRequestHandler(int statusCode, String reasonPhrase, String entityFormat) {
    this.statusCode = statusCode;
    this.reasonPhrase = reasonPhrase;
    this.entityFormat = entityFormat;
  }

  @Override
  public void handleRequest(HttpRequestContext requestContext, HttpResponseReadyCallback responseCallback) {
    String resolvedEntity = getResolvedEntity(requestContext.getRequest().getPath());
    responseCallback.responseReady(HttpResponse.builder()
        .statusCode(statusCode).reasonPhrase(reasonPhrase)
        .entity(new InputStreamHttpEntity(new ByteArrayInputStream(resolvedEntity.getBytes())))
        .addHeader(CONTENT_TYPE, TEXT.toRfcString()).build(),
                                   new ResponseStatusCallback() {

                                     @Override
                                     public void responseSendFailure(Throwable exception) {
                                       LOGGER.warn(format("Error while sending %s response %s", statusCode,
                                                          exception.getMessage()));
                                       if (LOGGER.isDebugEnabled()) {
                                         LOGGER.debug("exception thrown", exception);
                                       }
                                     }

                                     @Override
                                     public void responseSendSuccessfully() {}
                                   });
  }

  private String getResolvedEntity(String path) {
    return format(entityFormat, escapeHtml4(path));
  }

}
