/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.service.http.netty.impl.server.ChannelPromises.finishStreamingPromise;
import static org.mule.service.http.netty.impl.server.ChannelPromises.promiseToCallback;

import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.netty.impl.streaming.StatusCallback;

import java.io.IOException;
import java.io.InputStream;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandlerContext;

/**
 * Basic implementation of {@link ResponseSender}, that implements {@link #send()} as a template method. First, it saves the
 * header part of the response, and then delegates the content sending to the child implementations by calling
 * {@link #sendContent()}. The implementations of {@link #sendContent()} may call the protected methods of this class in order to
 * simplify their logic. When calling the sending methods, they will actually send the saved header part of the response if it
 * wasn't done yet. It's done this way so that if an exception is raised during the reading of the response content (within the
 * implementations of {@link #sendContent()}), we can send a corresponding failure response instead. Otherwise, if we send it as
 * soon as possible, we can't change the response's status code.
 */
public abstract class BaseResponseSender implements ResponseSender {

  protected final HttpRequest request;
  protected final HttpResponse response;

  private final ChannelHandlerContext ctx;
  private final StatusCallback statusCallback;
  private final HttpWriter writer;
  private boolean headerAlreadySent;

  protected BaseResponseSender(HttpRequest request, ChannelHandlerContext ctx, HttpResponse response,
                               StatusCallback statusCallback, HttpWriter writer) {
    this.writer = writer;
    checkArgument(statusCallback != null, "statusCallback can't be null");

    this.request = request;
    this.response = response;
    this.ctx = ctx;
    this.statusCallback = statusCallback;
  }

  @Override
  public void send() throws IOException {
    // The header part of the response will be sent later. It's done this way so that if an exception is raised
    // during the reading of the response CONTENT, we can send a corresponding failure response instead. Otherwise, if
    // we send it at this point, we can't change the response's status code.
    this.headerAlreadySent = false;
    sendContent();
  }

  protected void sendHeaderIfNeeded() {
    if (!headerAlreadySent) {
      writer.writeResponseHeader(request, response, promiseToCallback(ctx, statusCallback, false));
      headerAlreadySent = true;
    }
  }

  protected ByteBuf createBuffer(int size) {
    return ctx.alloc().buffer(size, size);
  }

  protected void sendLastContentAndFinishStreaming(ByteBuf content, InputStream contentAsInputStream) {
    sendHeaderIfNeeded();
    writer.writeContent(content, true, finishStreamingPromise(ctx, contentAsInputStream, statusCallback));
  }

  protected abstract void sendContent() throws IOException;
}
